type Vec2Integer
   x as integer
   y as integer
endtype
type Vec2Float
   x as float
   y as float
endtype
type Vec3Integer
   x as integer
   y as integer
   z as integer
endtype
type Vec3Float
   x as float
   y as float
   z as float
endtype
type Vec4Integer
   x as integer
   y as integer
   z as integer
   w as integer
endtype
type Vec4Float
   x as float
   y as float
   z as float
   w as float
endtype

type CameraData
  InUse as integer
  Position as Vec3Float
  Angle as Vec3Float
  Direction as Vec3Float
  Backdrop as Vec3Integer
  Aspect as float
  Range as Vec2Float
  Fov as float
  Ortho as Vec2Float
  View as Vec4Float
  Image as integer
  ImageDepth as integer
  ImageSize as Vec2Float
  Object as integer[]
  ObjectCount as integer
  ObjectFreeUp as integer
  ObjectTotal as integer
  ObjectRestack as integer
  Quad as integer
endtype

type EffectData
  InUse as integer
  ShaderCount as integer
  Technique as integer
  Shader as integer[]
  TechniqueName as string[]
  ObjectCount as integer
  Object as integer[]
  ObjectFreeUp as integer
  ObjectTotal as integer
  ObjectRestack as integer
  MeshCount as integer
  Mesh as integer[]
  MeshObject as integer[]
  MeshFreeUp as integer
  MeshTotal as integer
  MeshRestack as integer
endtype

`--------------
` Common Globals
`--------------
function InitCommonGlobals()
   global OldMouseX as float : OldMouseX=GetRawMouseX()
   global OldMouseY as float : OldMouseY=GetRawMouseX()
   global OldMouseZ as float : OldMouseZ=GetRawMouseWheel()
   global MouseMoveX as float
   global MouseMoveY as float
   global MinVec as Vec3Float
   global MaxVec as Vec3Float
   global MinMaxPos as Vec3Float
   global MinMaxSize as Vec3Float
   global Vector4F as Vec4Float
   global Vector3AGK as Vec4Integer
   global CameraCurrent as integer : CameraCurrent=-1
   global CamerasCount as integer : CamerasCount=-1
   global EffectsCount as integer : EffectsCount=-1
   global dim Cameras[] as CameraData
   global dim Effects[] as EffectData
   global dim Vector2Float[268] as Vec2Float
   global dim Vector3Float[268] as Vec3Float
   global dim Vector4Float[268] as Vec4Float
   global dim Matrix4Float[268,4] as Vec4Float
   global dim Matrix4X1Float[16] as float
   global dim Matrix4X2Float[16] as float
   global dim Matrix4X3Float[16] as float
   global dim Vector4Integer[32] as Vec4Integer
   global dim ShaderEffectDefineStr[] as string
   global dim ShaderEffectDefineVal[] as integer
   global dim ViewProjectVec[15] as float
   global dim ViewProjectFrustumVec[32,5,3] as float
   global dim AABBox[7] as Vec3Integer
   Vector3AGK.x=CreateVector3()
   Vector3AGK.y=CreateVector3()
   Vector3AGK.z=CreateVector3()
   Vector3AGK.w=CreateVector3()
   for b=0 to 7
      AABBox[b].x=1:AABBox[b].y=1:AABBox[b].z=1
   next b
   AABBox[1].x=-1:AABBox[2].y=-1:AABBox[3].x=-1:AABBox[3].y=-1
   AABBox[4].z=-1:AABBox[5].x=-1:AABBox[5].z=-1:AABBox[6].y=-1
   AABBox[6].z=-1:AABBox[7].x=-1:AABBox[7].y=-1:AABBox[7].z=-1
endfunction

`--------------
` Free Image
`--------------
function FreeUpImage(iFreeup as integer)
   repeat
      inc iFreeup
   until GetImageExists(iFreeup)=0
endfunction iFreeup

`--------------
` Free Object
`--------------
function FreeUpObject(iFreeup as integer)
   repeat
      inc iFreeup
   until GetObjectExists(iFreeup)=0
endfunction iFreeup

`--------------
` exp
`--------------
function Exp(iValue as float)
   r#=2.71828182846^iValue
endfunction r#

`--------------
` hSin
`--------------
function hSin(iValue as float)
   r#=iValue*0.01745329251
   r#=(Exp(r#)-Exp(-r#))/2
endfunction r#

`--------------
` hCos
`--------------
function hCos(iValue as float)
   r#=iValue*0.01745329251
   r#=(Exp(r#)+Exp(-r#))/2
endfunction r#

`--------------
` hTan
`--------------
function hTan(iValue as float)
   r#=hSin(iValue)/hCos(iValue)
endfunction r#

`--------------
` CurveValue
`--------------
function CurveValue(iDestination as float,iCurrent as float,iSpeed as float)
   r#=iCurrent+((iDestination-iCurrent)/iSpeed)
endfunction r#

`--------------
` CurveAngle
`--------------
function CurveAngle(iDestination as float,iCurrent as float,iSpeed as float)
   d#=wrapvalue(iCurrent-iDestination)
   if d#<=180.0 then d#=-d# else d#=360-d#
   r#=wrapvalue(iCurrent+(d#/iSpeed))
endfunction r#

`--------------
` WrapValue
`--------------
function WrapValue(iVal as float)
   r#=(iVal/360)
   r#=(r#-floor(r#))*360
endfunction r#

`--------------
` Linear Interpolation
`--------------
function Lerp(iA as float,iB as float,iValue as float)
   r#=iA+iValue*(iB-iA)
endfunction r#

`--------------
` Distance
`--------------
function Distance(iX as float,iY as float,iZ as float,ipX as float,ipY as float,ipZ as float)
   r#=sqrt((iX-ipX)^2+(iY-ipY)^2+(iZ-ipZ)^2)
endfunction r#

`--------------
` Text
`--------------
function Text(iX as integer,iY as integer,iStr as string)
   tText=CreateText(iStr)
   SetTextPosition(tText,iX,iY)
   SetTextSize(tText,20)
   SetTextbold(tText,1)
   SetTextColor(tText,255,255,255,255)
endfunction
function CenterText(iX as integer,iY as integer,iStr as string)
   tText=CreateText(iStr)
   SetTextPosition(tText,iX-(GetTextTotalWidth(tText)*2.5),iY)
   SetTextSize(tText,20)
   SetTextbold(tText,1)
   SetTextColor(tText,255,255,255,255)
endfunction

`--------------
` Load Texture
`--------------
function LoadTexture(iImage as integer,iFile as string)
   if GetImageExists(iImage)=1 then DeleteImage(iImage)
   SetGenerateMipmaps(1)
   loadimage(iImage,iFile)
   SetImageWrapU(iImage,1)
   SetImageWrapV(iImage,1)
endfunction

`--------------
` MouseMove
`--------------
function MouseMoveX()
   if GetRawMouseX()<>OldMouseX
      MouseMoveX=-(OldMouseX-GetRawMouseX())
      OldMouseX=GetRawMouseX()
   endif
endfunction MouseMoveX
function MouseMoveY()
   if GetRawMouseY()<>OldMouseY
      MouseMoveY=-(OldMouseY-GetRawMouseY())
      OldMouseY=GetRawMouseY()
   endif
endfunction MouseMoveY
function MouseMoveZ()
   if GetRawMouseWheel()<>OldMouseZ
      MouseMoveZ=-(OldMouseZ-GetRawMouseWheel())*10
      OldMouseZ=GetRawMouseWheel()
   endif
endfunction MouseMoveZ
function MouseMoveReset()
   MouseMoveX=0 : MouseMoveY=0 : MouseMoveZ=0
   SetRawMousePosition(GetVirtualWidth()/2,GetVirtualHeight()/2)
   OldMouseX=GetRawMouseX()
   OldMouseY=GetRawMouseY()
   OldMouseZ=GetRawMouseWheel()
endfunction

`--------------
` Create Camera
`--------------
function CreateCamera()
   tCameraID=-1
   for c=0 to CamerasCount
      if Cameras[c].InUse=0
         tCameraID=c:exit
      endif
   next c
   if tCameraID=-1
      inc CamerasCount
      tCameraID=CamerasCount
      Cameras.length=Cameras.length+1
   endif
   sw#=GetVirtualWidth() : sh#=GetVirtualHeight()
   Cameras[tCameraID].InUse=1
   Cameras[tCameraID].Image=0
   Cameras[tCameraID].ImageDepth=-1
   Cameras[tCameraID].Position.x=0 : Cameras[tCameraID].Position.y=0 : Cameras[tCameraID].Position.z=0
   Cameras[tCameraID].Angle.x=0 : Cameras[tCameraID].Angle.y=0 : Cameras[tCameraID].Angle.z=0
   Cameras[tCameraID].Direction.x=0 : Cameras[tCameraID].Direction.y=0 : Cameras[tCameraID].Direction.z=1
   Cameras[tCameraID].Backdrop.x=64 : Cameras[tCameraID].Backdrop.y=64 : Cameras[tCameraID].Backdrop.z=64
   Cameras[tCameraID].Aspect=sw#/sh#
   Cameras[tCameraID].Range.x=1 : Cameras[tCameraID].Range.y=64000
   Cameras[tCameraID].Fov=125
   Cameras[tCameraID].Ortho.x=0 : Cameras[tCameraID].Ortho.y=0
   Cameras[tCameraID].View.x=0 : Cameras[tCameraID].View.y=0 : Cameras[tCameraID].View.z=sw# : Cameras[tCameraID].View.w=sh#
   Cameras[tCameraID].ImageSize.x=0 : Cameras[tCameraID].ImageSize.y=0
   Cameras[tCameraID].Quad=0
   Cameras[tCameraID].ObjectCount=-1
   Cameras[tCameraID].Object.length=0
   Cameras[tCameraID].ObjectFreeUp=0
   Cameras[tCameraID].ObjectTotal=-1
   Cameras[tCameraID].ObjectRestack=0
endfunction tCameraID

`--------------
` Camera Exist
`--------------
function CameraExist(iCamera as integer)
   if iCamera<CamerasCount+1 and iCamera>-1
      if Cameras[iCamera].InUse=1 then exitfunction 1
   endif
endfunction 0

`--------------
` Delete Camera
`--------------
function DeleteCamera(iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction
   Cameras[iCamera].InUse=0
   if Cameras[iCamera].Quad>0
      if GetObjectExists(Cameras[iCamera].Quad)=1 then DeleteObject(Cameras[iCamera].Quad)
   endif
   Cameras[iCamera].Object.length=0
endfunction

`--------------
` Set Current Camera
`--------------
function CurrentCamera(iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction
   if CameraCurrent<>iCamera
      CameraCurrent=iCamera
      sw#=GetDevicewidth() : sh#=GetDeviceHeight()
      SetClearColor(Cameras[iCamera].Backdrop.x,Cameras[iCamera].Backdrop.y,Cameras[iCamera].Backdrop.z)
      if Cameras[iCamera].Image>0 or Cameras[iCamera].ImageDepth>0
         SetRenderToImage(Cameras[iCamera].Image,Cameras[iCamera].ImageDepth)
         if Cameras[iCamera].Quad=0 then ClearScreen()
         tw#=Cameras[iCamera].ImageSize.x
         th#=Cameras[iCamera].ImageSize.y
      else
         SetRenderToScreen()
         tw#=sw# : th#=sh#
      endif
      SetCameraPosition(1,Cameras[iCamera].Position.x,Cameras[iCamera].Position.y,Cameras[iCamera].Position.z)
      SetCameraRotation(1,Cameras[iCamera].Angle.x,Cameras[iCamera].Angle.y,Cameras[iCamera].Angle.z)
      SetCameraRange(1,Cameras[iCamera].Range.x,Cameras[iCamera].Range.y)
      if Cameras[iCamera].Ortho.x=0 and Cameras[iCamera].Ortho.y=0
         SetCameraFov(1,Cameras[iCamera].Fov) 
         SetCameraAspect(1,Cameras[iCamera].Aspect)
      else
        if Cameras[iCamera].Ortho.x>Cameras[iCamera].Ortho.y
           a#=Cameras[iCamera].Ortho.y : w#=Cameras[iCamera].Ortho.x
        else
           a#=Cameras[iCamera].Ortho.x : w#=Cameras[iCamera].Ortho.y
        endif
        SetCameraFov(1,0)
        SetCameraAspect(1,w#/a#)
        SetCameraOrthoWidth(1,w#*0.5)
      endif
      if Cameras[iCamera].View.x>0 or Cameras[iCamera].View.y>0 or Cameras[iCamera].View.z<tw# or Cameras[iCamera].View.w<th#
         Left#=Cameras[iCamera].View.x/tw#
         Right#=Cameras[iCamera].View.z/tw#
         Top#=Cameras[iCamera].View.y/th#
         Bottom#=Cameras[iCamera].View.w/th#
         SetScissor(Left#*sw#,Top#*sh#,Right#*sw#,Bottom#*sh#)
         SetCameraOffCenter(1,1)
         sx#=1.0/(Right#-Left#)
         sy#=1.0/(Bottom#-Top#)
         cx#=(0.5-(Left#+((Right#-Left#)*0.5)))*(sx#*2)
         cy#=-(0.5-(Top#+((Bottom#-Top#)*0.5)))*(sy#*2)
         SetCameraBounds(1,-sx#+cx#,sx#+cx#,sy#+cy#,-sy#+cy#)
      else
         SetScissor(0,0,0,0)
         SetCameraOffCenter(1,0)
      endif
   endif
endfunction

`--------------
` Render Camera
`--------------
function RenderCamera(iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction
   if CameraCurrent<>iCamera then CurrentCamera(iCamera)
   if Cameras[CameraCurrent].Quad>0 
      if GetObjectexists(Cameras[CameraCurrent].Quad)=1
         drawobject(Cameras[CameraCurrent].Quad)
      else
         Cameras[CameraCurrent].Quad=0 
      endif
   endif 
   for o=0 to Cameras[CameraCurrent].ObjectTotal
      if Cameras[CameraCurrent].Object[o]>0
         if GetObjectexists(Cameras[CameraCurrent].Object[o])=1 
            drawobject(Cameras[CameraCurrent].Object[o])
         else
            if Cameras[CameraCurrent].ObjectFreeUp>o then Cameras[CameraCurrent].ObjectFreeUp=o
            Cameras[CameraCurrent].Object[o]=0 : inc Cameras[CameraCurrent].ObjectRestack
         endif
      endif
   next o
endfunction

`--------------
` Remove Object From Camera
`--------------
function RemoveObjectFromCamera(iObject as integer,iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction
   for o=0 to Cameras[iCamera].ObjectCount
      if Cameras[iCamera].Object[o]=iObject
         if Cameras[iCamera].ObjectFreeUp>o then Cameras[iCamera].ObjectFreeUp=o
         Cameras[iCamera].Object[o]=0 : inc Cameras[iCamera].ObjectRestack
      endif
   next o
   tChildren=GetObjectNumChildren(iObject)
   if tChildren>0
      for c=1 to tChildren
         tObject=GetObjectChildID(iObject,c)
         for o=0 to Cameras[iCamera].ObjectCount
            if Cameras[iCamera].Object[o]=tObject
               if Cameras[iCamera].ObjectFreeUp>o then Cameras[iCamera].ObjectFreeUp=o
               Cameras[iCamera].Object[o]=0 : inc Cameras[iCamera].ObjectRestack
            endif
         next o
      next c
      CameraRestackObjectArray(iCamera)
   endif
endfunction
// Restack Camera Objects array to reduce fragmentation
function CameraRestackObjectArray(iCamera as integer)
   if Cameras[iCamera].ObjectRestack>25
      Cameras[iCamera].ObjectRestack=0
      tNextEelement=Cameras[iCamera].ObjectFreeUp
      tCurrEelement=Cameras[iCamera].ObjectFreeUp
      repeat
         if Cameras[iCamera].Object[tNextEelement]>0
            Cameras[iCamera].Object[tCurrEelement]=Cameras[iCamera].Object[tNextEelement]
            if tNextEelement>tCurrEelement then Cameras[iCamera].Object[tNextEelement]=0
            inc tCurrEelement
         endif
         inc tNextEelement
      until tNextEelement>Cameras[iCamera].ObjectTotal
      Cameras[iCamera].ObjectTotal=tCurrEelement
      Cameras[iCamera].ObjectFreeUp=tCurrEelement
   endif
endfunction

`--------------
` Draw Object To Camera
`--------------
function DrawObjectToCamera(iObject as integer,iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction
   RemoveObjectFromCamera(iObject,iCamera)
   if GetObjectexists(iObject)=1
      tObject=-1
      for o=Cameras[iCamera].ObjectFreeUp to Cameras[iCamera].ObjectCount
         if Cameras[iCamera].Object[o]=0
            tObject=o:exit
         endif
      next o
      if tObject=-1
         inc Cameras[iCamera].ObjectCount
         Cameras[iCamera].Object.length=Cameras[iCamera].ObjectCount+1
         tObject=Cameras[iCamera].ObjectCount
      endif
      Cameras[iCamera].ObjectFreeUp=tObject
      if tObject>Cameras[iCamera].ObjectTotal then Cameras[iCamera].ObjectTotal=tObject
      Cameras[iCamera].Object[tObject]=iObject
      tChildren=GetObjectNumChildren(iObject)
      if tChildren>0
         for c=1 to tChildren
             tObject=GetObjectChildID(iObject,c)
             tObject=-1
             for o=Cameras[iCamera].ObjectFreeUp to Cameras[iCamera].ObjectCount
                if Cameras[iCamera].Object[o]=0
                   tObject=o:exit
                endif
             next o
            if tObject=-1
               inc Cameras[iCamera].ObjectCount
               Cameras[iCamera].Object.length=Cameras[iCamera].ObjectCount+1
               tObject=Cameras[iCamera].ObjectCount
            endif
            Cameras[iCamera].ObjectFreeUp=tObject
            if tObject>Cameras[iCamera].ObjectTotal then Cameras[iCamera].ObjectTotal=tObject
            Cameras[iCamera].Object[tObject]=tObject
         next c
      endif
   endif
endfunction

`--------------
` Camera To Image
`--------------
function CameraToImage(iCamera as integer,iImage as integer,iDepthId as integer,iWidth as integer,iHeight as integer)
   if CameraExist(iCamera)=0 then exitfunction
   if iImage>0
      if GetImageExists(iImage)=1 then DeleteImage(iImage)
      CreateRenderImage(iImage,iWidth,iHeight,0,0) 
      SetImageWrapU(iImage,0) : SetImageWrapV(iImage,0)
      Cameras[iCamera].Image=iImage
   endif
   if iDepthId>0
      if GetImageExists(iDepthId)=1 then DeleteImage(iDepthId)
      CreateRenderImage(iDepthId,iWidth,iHeight,1,0) 
      SetImageWrapU(iDepthId,0) : SetImageWrapV(iDepthId,0)
      Cameras[iCamera].ImageDepth=iDepthId
   endif
   if iImage>0 or iDepthId>0 then CameraView(iCamera,0,0,iWidth,iHeight)
   if CameraCurrent=iCamera then SetRenderToImage(Cameras[iCamera].Image,Cameras[iCamera].ImageDepth)
endfunction

`--------------
` Camera To Render Target
`--------------
function CameraToRenderTarget(iCamera as integer,iObject as integer)
   if CameraExist(iCamera)=0 then exitfunction
   Cameras[iCamera].Position.x=0 : Cameras[iCamera].Position.y=-999999 : Cameras[iCamera].Position.z=0
   Cameras[iCamera].Angle.x=0 : Cameras[iCamera].Angle.y=0 : Cameras[iCamera].Angle.z=0
   Cameras[iCamera].Direction.x=0 : Cameras[iCamera].Direction.y=0 : Cameras[iCamera].Direction.z=1
   Cameras[iCamera].Backdrop.x=0 : Cameras[iCamera].Backdrop.y=0 : Cameras[iCamera].Backdrop.z=0
   Cameras[iCamera].Aspect=1
   Cameras[iCamera].Range.x=1 : Cameras[iCamera].Range.y=2
   Cameras[iCamera].Fov=90
   Cameras[iCamera].Ortho.x=0 : Cameras[iCamera].Ortho.y=0
   if GetObjectExists(iObject)=1 then DeleteObject(iObject)
   if Cameras[iCamera].Quad>0
      if GetObjectExists(Cameras[iCamera].Quad)=1 then DeleteObject(Cameras[iCamera].Quad)
   endif
   CreateObjectQuad(iObject)
   Cameras[iCamera].Quad=iObject
endfunction

`--------------
` Color Backdrop
`--------------
function ColorBackdrop(iCamera as integer,iRed as float,iGreen as float,iBlue as float)
   if CameraExist(iCamera)=0 then exitfunction
   Cameras[iCamera].Backdrop.x=iRed : Cameras[iCamera].Backdrop.y=iGreen : Cameras[iCamera].Backdrop.z=iBlue
   if CameraCurrent=iCamera then SetClearColor(Cameras[iCamera].Backdrop.x,Cameras[iCamera].Backdrop.y,Cameras[iCamera].Backdrop.z)
endfunction

`--------------
` Set Camera Range
`--------------
function CameraRange(iCamera as integer,iNear as float,iFar as float)
   if CameraExist(iCamera)=0 then exitfunction
   Cameras[iCamera].Range.x=iNear : Cameras[iCamera].Range.y=iFar
   if CameraCurrent=iCamera then SetCameraRange(1,Cameras[iCamera].Range.x,Cameras[iCamera].Range.y)
endfunction

`--------------
` Set Camera Aspect
`--------------
function CameraAspect(iCamera as integer,iAspect as float)
   if CameraExist(iCamera)=0 then exitfunction
   Cameras[iCamera].Aspect=iAspect
   if CameraCurrent=iCamera 
      if Cameras[iCamera].Ortho.x=0 and Cameras[iCamera].Ortho.y=0 then SetCameraAspect(1,Cameras[iCamera].Aspect)
   endif
endfunction

`--------------
` Set Camera Fov
`--------------
function CameraFov(iCamera as integer,iFov as float)
   if CameraExist(iCamera)=0 then exitfunction
   Cameras[iCamera].Fov=iFov
   if CameraCurrent=iCamera 
      if Cameras[iCamera].Ortho.x=0 and Cameras[iCamera].Ortho.y=0 then SetCameraFov(1,Cameras[iCamera].Fov)
   endif
endfunction

`--------------
` Set Camera Ortho
`--------------
function CameraOrtho(iCamera as integer,iWidth as float,iHeight as float)
   if CameraExist(iCamera)=0 then exitfunction
   Cameras[iCamera].Ortho.x=iWidth : Cameras[iCamera].Ortho.y=iHeight
   if CameraCurrent=iCamera
      if Cameras[iCamera].Ortho.x=0 and Cameras[iCamera].Ortho.y=0 
         SetCameraAspect(1,Cameras[iCamera].Aspect)
         SetCameraFov(1,Cameras[iCamera].Fov)
      else
        if Cameras[iCamera].Ortho.x>Cameras[iCamera].Ortho.y
           a#=Cameras[iCamera].Ortho.y : w#=Cameras[iCamera].Ortho.x
        else
           a#=Cameras[iCamera].Ortho.x : w#=Cameras[iCamera].Ortho.y
        endif
        SetCameraAspect(1,w#/a#)
        SetCameraFov(1,0)
        SetCameraOrthoWidth(1,w#*0.5)
      endif
   endif
endfunction

`--------------
` Set Camera View
`--------------
function CameraView(iCamera as integer,iLeft as float,iTop as float,iRight as float,iBottom as float)
   if CameraExist(iCamera)=0 then exitfunction
   Cameras[iCamera].View.x=iLeft : Cameras[iCamera].View.y=iTop : Cameras[iCamera].View.z=iRight : Cameras[iCamera].View.w=iBottom
   if Cameras[iCamera].View.x<0 then Cameras[iCamera].View.x=0
   if Cameras[iCamera].View.y<0 then Cameras[iCamera].View.y=0
   sw#=GetDevicewidth() : sh#=GetDeviceHeight()
   tw#=sw# : th#=sh#
   if Cameras[iCamera].Image>0 or Cameras[iCamera].ImageDepth>0
      tw#=Cameras[iCamera].ImageSize.x
      th#=Cameras[iCamera].ImageSize.y
   endif
   if Cameras[iCamera].View.z>tw# then Cameras[iCamera].View.z=tw#
   if Cameras[iCamera].View.w>th# then Cameras[iCamera].View.w=th#
   if CameraCurrent=iCamera
      if Cameras[iCamera].View.x>0 or Cameras[iCamera].View.y>0 or Cameras[iCamera].View.z<tw# or Cameras[iCamera].View.w<th#
         Left#=Cameras[iCamera].View.x/tw#
         Right#=Cameras[iCamera].View.z/tw#
         Top#=Cameras[iCamera].View.y/th#
         Bottom#=Cameras[iCamera].View.w/th#
         SetScissor(Left#*sw#,Top#*sh#,Right#*sw#,Bottom#*sh#)
         SetCameraOffCenter(1,1)
         sx#=1.0/(Right#-Left#)
         sy#=1.0/(Bottom#-Top#)
         cx#=(0.5-(Left#+((Right#-Left#)*0.5)))*(sx#*2)
         cy#=-(0.5-(Top#+((Bottom#-Top#)*0.5)))*(sy#*2)
         SetCameraBounds(1,-sx#+cx#,sx#+cx#,sy#+cy#,-sy#+cy#)
      else
         SetScissor(0,0,0,0)
         SetCameraOffCenter(1,0)
      endif
   endif
endfunction

`--------------
` Set Camera Position
`--------------
function PositionCamera(iCamera as integer,iPosX as float,iPosY as float,iPosZ as float)
   if CameraExist(iCamera)=0 then exitfunction
   Cameras[iCamera].Position.x=iPosX: Cameras[iCamera].Position.y=iPosY : Cameras[iCamera].Position.z=iPosZ
   if CameraCurrent=iCamera then SetCameraPosition(1,Cameras[iCamera].Position.x,Cameras[iCamera].Position.y,Cameras[iCamera].Position.z)
endfunction

`--------------
` Set Camera Rotation
`--------------
function RotateCamera(iCamera as integer,iAngX as float,iAngY as float,iAngZ as float)
   if CameraExist(iCamera)=0 then exitfunction
   Cameras[iCamera].Angle.x=iAngX : Cameras[iCamera].Angle.y=iAngY : Cameras[iCamera].Angle.z=iAngZ
   RotateX_Matrix4(261,WrapValue(Cameras[iCamera].Angle.x)*0.01745329251)
   RotateY_Matrix4(262,WrapValue(Cameras[iCamera].Angle.y)*0.01745329251)
   RotateZ_Matrix4(263,WrapValue(Cameras[iCamera].Angle.z)*0.01745329251)
   Multiply_Matrix4(264,261,262)
   Multiply_Matrix4(264,264,263)
   Set_Vector3(261,0,0,1)
   Transform_Normals_Vector3(261,261,264)
   Cameras[iCamera].Direction.x=Vector3Float[261].x
   Cameras[iCamera].Direction.y=Vector3Float[261].y
   Cameras[iCamera].Direction.z=Vector3Float[261].z
   if CameraCurrent=iCamera then SetCameraRotation(1,Cameras[iCamera].Angle.x,Cameras[iCamera].Angle.y,Cameras[iCamera].Angle.z)
endfunction

`--------------
` Point Camera
`--------------
function PointCamera(iCamera as integer,iPosX as float,iPosY as float,iPosZ as float)
   if CameraExist(iCamera)=0 then exitfunction
   Set_Vector3(261,iPosX-Cameras[iCamera].Position.x,iPosY-Cameras[iCamera].Position.y,iPosZ-Cameras[iCamera].Position.z)
   Normalize_Vector3(261,261)
   RotateCamera(iCamera ,asin(-Vector3Float[261].y),atanfull(Vector3Float[261].x,-Vector3Float[261].z),0)
endfunction

`--------------
` Move Camera 
`--------------
function MoveCameraX(iCamera as integer,iValue as float)
   if CameraExist(iCamera)=0 then exitfunction
   RotateX_Matrix4(261,WrapValue(Cameras[iCamera].Angle.x)*0.01745329251)
   RotateY_Matrix4(262,WrapValue(Cameras[iCamera].Angle.y)*0.01745329251)
   RotateZ_Matrix4(263,WrapValue(Cameras[iCamera].Angle.z)*0.01745329251)
   Multiply_Matrix4(264,261,262)
   Multiply_Matrix4(264,264,263)
   Set_Vector3(261,1,0,0)
   Transform_Normals_Vector3(261,261,264)
   PositionCamera(iCamera,Cameras[iCamera].Position.x+(Vector3Float[261].x*iValue),Cameras[iCamera].Position.y+(Vector3Float[261].y*iValue),Cameras[iCamera].Position.z+(Vector3Float[261].z*iValue))
endfunction
function MoveCameraY(iCamera as integer,iValue as float)
   if CameraExist(iCamera)=0 then exitfunction
   RotateX_Matrix4(261,WrapValue(Cameras[iCamera].Angle.x)*0.01745329251)
   RotateY_Matrix4(262,WrapValue(Cameras[iCamera].Angle.y)*0.01745329251)
   RotateZ_Matrix4(263,WrapValue(Cameras[iCamera].Angle.z)*0.01745329251)
   Multiply_Matrix4(264,261,262)
   Multiply_Matrix4(264,264,263)
   Set_Vector3(261,0,1,0)
   Transform_Normals_Vector3(261,261,264)
   PositionCamera(iCamera,Cameras[iCamera].Position.x+(Vector3Float[261].x*iValue),Cameras[iCamera].Position.y+(Vector3Float[261].y*iValue),Cameras[iCamera].Position.z+(Vector3Float[261].z*iValue))
endfunction
function MoveCameraZ(iCamera as integer,iValue as float)
   if CameraExist(iCamera)=0 then exitfunction
   PositionCamera(iCamera,Cameras[iCamera].Position.x+(Cameras[iCamera].Direction.x*iValue),Cameras[iCamera].Position.y+(Cameras[iCamera].Direction.y*iValue),Cameras[iCamera].Position.z+(Cameras[iCamera].Direction.z*iValue))
endfunction

`--------------
` Copy Camera
`--------------
function CopyCamera(iCamera as integer,iCameraSorce as integer)
   if CameraExist(iCamera)=0 then exitfunction
   if CameraExist(iCameraSorce)=0 then exitfunction
   Cameras[iCamera].Position.x=Cameras[iCameraSorce].Position.x : Cameras[iCamera].Position.y=Cameras[iCameraSorce].Position.y : Cameras[iCamera].Position.z=Cameras[iCameraSorce].Position.z
   Cameras[iCamera].Angle.x=Cameras[iCameraSorce].Angle.x : Cameras[iCamera].Angle.y=Cameras[iCameraSorce].Angle.y : Cameras[iCamera].Angle.z=Cameras[iCameraSorce].Angle.z
   Cameras[iCamera].Direction.x=Cameras[iCameraSorce].Direction.x : Cameras[iCamera].Direction.y=Cameras[iCameraSorce].Direction.y : Cameras[iCamera].Direction.z=Cameras[iCameraSorce].Direction.z
   Cameras[iCamera].Aspect=Cameras[iCameraSorce].Aspect
   Cameras[iCamera].Range.x=Cameras[iCameraSorce].Range.x : Cameras[iCamera].Range.y=Cameras[iCameraSorce].Range.y
   Cameras[iCamera].Fov=Cameras[iCameraSorce].Fov
   Cameras[iCamera].Ortho.x=Cameras[iCameraSorce].Ortho.x : Cameras[iCamera].Ortho.y=Cameras[iCameraSorce].Ortho.y
   if CameraCurrent=iCamera 
      CameraCurrent=iCamera+1
      CurrentCamera(iCamera)
   endif
endfunction

`--------------
` View Matrix
`--------------
function ViewMatrix(iCamera as integer,iMatrixResult as integer)
   if CameraExist(iCamera)=0 then exitfunction
   CX#=Cos(Cameras[iCamera].Angle.x)
   SX#=Sin(Cameras[iCamera].Angle.x)
   CY#=Cos(Cameras[iCamera].Angle.y)
   SY#=Sin(Cameras[iCamera].Angle.y)
   CZ#=Cos(Cameras[iCamera].Angle.z)
   SZ#=Sin(Cameras[iCamera].Angle.z)
   Set_Vector3(260,0,0,0)
   Set_Vector3(261,-1.0*SZ#*(-1.0*SX#)+CZ#*SY#*CX#,CZ#*(-1.0*SX#)+SZ#*SY#*CX#,CY#*CX#)
   Set_Vector3(262,-1.0*SZ#*CX#+CZ#*SY#*SX#,CZ#*CX#+SZ#*SY#*SX#,CY#*SX#)
   Build_LookAtLH_Matrix4(260,260,261,262)
   Translate_Matrix4(261,-Cameras[iCamera].Position.x,-Cameras[iCamera].Position.y,-Cameras[iCamera].Position.z)
   Multiply_Matrix4(iMatrixResult,261,260)
endfunction

`--------------
` Projection Matrix
`--------------
function ProjectionMatrix(iCamera as integer,iMatrixResult as integer)
   if CameraExist(iCamera)=0 then exitfunction
   if Cameras[iCamera].Ortho.x=0 and Cameras[iCamera].Ortho.y=0 
      FOV#=2.0*atan(tan(Cameras[iCamera].Fov*0.5)/Cameras[iCamera].Aspect)
      FOV#=1.0/tan(FOV#/2.0)
      Fill_Matrix4(iMatrixResult,0.0)
      Matrix4Float[iMatrixResult,1].x=FOV#/Cameras[iCamera].Aspect
      Matrix4Float[iMatrixResult,2].y=-FOV#
      Matrix4Float[iMatrixResult,3].z=-1.0/(Cameras[iCamera].Range.y-Cameras[iCamera].Range.x)
      Matrix4Float[iMatrixResult,3].w=1.0
      Matrix4Float[iMatrixResult,4].z=(Cameras[iCamera].Range.y-Cameras[iCamera].Range.x)/Cameras[iCamera].Range.y
   else
      Build_OrthoLH_Matrix4(iMatrixResult,Cameras[iCamera].Ortho.x,Cameras[iCamera].Ortho.y,Cameras[iCamera].Range.x,Cameras[iCamera].Range.y)
   endif
endfunction

`--------------
` Get Camera Position 
`--------------
function CameraPositionX(iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction 0
   r#=Cameras[iCamera].Position.x
endfunction r#
function CameraPositionY(iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction 0
   r#=Cameras[iCamera].Position.y
endfunction r#
function CameraPositionZ(iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction 0
   r#=Cameras[iCamera].Position.z
endfunction r#

`--------------
` Get Camera angle 
`--------------
function CameraAngleX(iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction 0
   r#=Cameras[iCamera].Angle.x
endfunction r#
function CameraAngleY(iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction 0
   r#=Cameras[iCamera].Angle.y
endfunction r#
function CameraAngleZ(iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction 0
   r#=Cameras[iCamera].Angle.z
endfunction r#

`--------------
` Get Camera Direction 
`--------------
function CameraDirectionX(iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction 0
   r#=Cameras[iCamera].Direction.x
endfunction r#
function CameraDirectionY(iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction 0
   r#=Cameras[iCamera].Direction.y
endfunction r#
function CameraDirectionZ(iCamera as integer)
   if CameraExist(iCamera)=0 then exitfunction 0
   r#=Cameras[iCamera].Direction.z
endfunction r#

`--------------
` Create Effect
`--------------
function CreateEffect()
   tEffectID=-1
   for e=0 to EffectsCount
      if Effects[e].InUse=0
         tEffectID=e:exit
      endif
   next e
   if tEffectID=-1
      inc EffectsCount
      tEffectID=EffectsCount
      Effects.length=Effects.length+1
   endif
   Effects[tEffectID].InUse=1
   Effects[tEffectID].Technique=0
   Effects[tEffectID].ShaderCount=-1
   Effects[tEffectID].ObjectCount=-1
   Effects[tEffectID].ObjectFreeUp=0
   Effects[tEffectID].ObjectTotal=-1
   Effects[tEffectID].ObjectRestack=0
   Effects[tEffectID].MeshCount=-1
   Effects[tEffectID].MeshFreeUp=0
   Effects[tEffectID].MeshTotal=-1
   Effects[tEffectID].MeshRestack=0
endfunction tEffectID

`--------------
` Effect Exist
`--------------
function GetEffectExist(iEffect as integer)
   if iEffect<EffectsCount+1 and iEffect>-1
      if Effects[iEffect].InUse=1 then exitfunction 1
   endif
endfunction 0

`--------------
` Delete Effect
`--------------
function DeleteEffect(iEffect as integer)
   if GetEffectExist(iEffect)=0 then exitfunction
   Effects[iEffect].InUse=0
   for o=0 to Effects[iEffect].ObjectCount
      if Effects[iEffect].Object[o]>0
         if GetObjectexists(Effects[iEffect].Object[o])=1 then setobjectshader(Effects[iEffect].Object[o],0)
      endif
   next o
   for m=0 to Effects[iEffect].MeshCount
      if Effects[iEffect].Mesh[m]>0
         if GetObjectexists(Effects[iEffect].MeshObject[m])=1 then setobjectmeshshader(Effects[iEffect].MeshObject[m],Effects[iEffect].Mesh[m],0)
      endif
   next m
   for s=0 to Effects[iEffect].ShaderCount
     deleteShader(Effects[iEffect].Shader[s])
   next s
   Effects[iEffect].Shader.length=0
   Effects[iEffect].TechniqueName.length=0
   Effects[iEffect].Object.length=0
   Effects[iEffect].Mesh.length=0
   Effects[iEffect].MeshObject.length=0
endfunction

`--------------
` Set Shader Effect Define (preprocessor directives)
`--------------
function SetShaderEffectDefine(iDefine as string,iValue as integer)
   ShaderEffectDefineStr.length=ShaderEffectDefineStr.length+1
   ShaderEffectDefineVal.length=ShaderEffectDefineVal.length+1
   tIdx=ShaderEffectDefineStr.length
   ShaderEffectDefineStr[tIdx]=iDefine
   ShaderEffectDefineVal[tIdx]=iValue
endfunction

`--------------
` Load Effect Technique
`--------------
function LoadEffectTechnique(iEffect as integer,iTechniqueName as string,iVSFile as string,iPSFile as string)
   if GetEffectExist(iEffect)=0 then exitfunction
   if GetFileExists(iVSFile)=1 and GetFileExists(iPSFile)=1
      inc Effects[iEffect].ShaderCount
      Effects[iEffect].Shader.length=Effects[iEffect].ShaderCount+1
      Effects[iEffect].TechniqueName.length=Effects[iEffect].ShaderCount+1
      tShaderCount=Effects[iEffect].ShaderCount
      Effects[iEffect].TechniqueName[tShaderCount]=lower(iTechniqueName)
      tDefineCount=ShaderEffectDefineVal.length
      if tDefineCount=-1
         Effects[iEffect].Shader[tShaderCount]=LoadShader(iVSFile,iPSFile)
      else
         local tVertexShaderStr as string
         tFile=OpenToRead(iVSFile)
         repeat
            tStr$=ReadLine(tFile)
            if FindString(tStr$,"#define")=1
               for x=0 to tDefineCount
                  tFind=FindString(tStr$,ShaderEffectDefineStr[x],1,-1)
                  if tFind>0 then tStr$=Left(tStr$,tFind+len(ShaderEffectDefineStr[x]))+str(ShaderEffectDefineVal[X])
               next x
            endif
            tVertexShaderStr=tVertexShaderStr+tStr$+chr(10)
         until GetFilePos(tFile)>GetFileSize(tFile)-1
         CloseFile(tFile)
         local tPixelShaderStr as string
         tFile=OpenToRead(iPSFile)
         repeat
            tStr$=ReadLine(tFile)
            if FindString(tStr$,"#define")=1
               for x=0 to tDefineCount
                  tFind=FindString(tStr$,ShaderEffectDefineStr[x],1,-1)
                  if tFind>0 then tStr$=Left(tStr$,tFind+len(ShaderEffectDefineStr[x]))+str(ShaderEffectDefineVal[X])
               next x
            endif
            tPixelShaderStr=tPixelShaderStr+tStr$+chr(10)
         until GetFilePos(tFile)>GetFileSize(tFile)-1
         CloseFile(tFile)
         Effects[iEffect].Shader[tShaderCount]=LoadShaderFromString(tVertexShaderStr,tPixelShaderStr)
      endif
   else
      if iVSFile="" and GetFileExists(iPSFile)=1
         inc Effects[iEffect].ShaderCount
         Effects[iEffect].Shader.length=Effects[iEffect].ShaderCount+1
         Effects[iEffect].TechniqueName.length=Effects[iEffect].ShaderCount+1
         tShaderCount=Effects[iEffect].ShaderCount
         Effects[iEffect].TechniqueName[tShaderCount]=lower(iTechniqueName)
         Effects[iEffect].Shader[tShaderCount]=LoadFullScreenShader(iPSFile)
      endif
   endif
   ShaderEffectDefineStr.length=-1
   ShaderEffectDefineVal.length=-1
endfunction

`--------------
` Remove Object Effect
`--------------
function RemoveObjectEffect(iObject as integer)
   if GetObjectexists(iObject)=1 then SetObjectShader(iObject,0)
   for e=0 to EffectsCount
      for o=0 to Effects[e].ObjectTotal
         if Effects[e].Object[o]=iObject
            if Effects[e].ObjectFreeUp>o then Effects[e].ObjectFreeUp=o
            Effects[e].Object[o]=0 : inc Effects[e].ObjectRestack
         endif
      next o
      for m=0 to Effects[e].MeshCount
         if Effects[e].MeshObject[m]=iObject
            if Effects[e].MeshFreeUp>m then Effects[e].MeshFreeUp=m
            Effects[e].Mesh[m]=0 : inc Effects[e].MeshRestack
            Effects[e].MeshObject[m]=0
         endif
      next m
      EffectRestackObjectArray(e)
      EffectRestackMeshArray(e)
   next e
endfunction
// Restack Effects Objects array to reduce fragmentation
function EffectRestackObjectArray(iEffect as integer)
   if Effects[iEffect].ObjectRestack>25
      Effects[iEffect].ObjectRestack=0
      tNextEelement=Effects[iEffect].ObjectFreeUp
      tCurrEelement=Effects[iEffect].ObjectFreeUp
      repeat
         if Effects[iEffect].Object[tNextEelement]>0
            Effects[iEffect].Object[tCurrEelement]=Effects[iEffect].Object[tNextEelement]
            if tNextEelement>tCurrEelement then Effects[iEffect].Object[tNextEelement]=0
            inc tCurrEelement
         endif
         inc tNextEelement
      until tNextEelement>Effects[iEffect].ObjectTotal
      Effects[iEffect].ObjectTotal=tCurrEelement
      Effects[iEffect].ObjectFreeUp=tCurrEelement
   endif
endfunction

`--------------
` Set Object Effect
`--------------
function SetObjectEffect(iObject as integer,iEffect as integer,iTangents as integer)
   RemoveObjectEffect(iObject)
   if GetEffectExist(iEffect)=0 then exitfunction
   if GetObjectexists(iObject)=1
      if iTangents=1
         for m=1 to GetObjectNumMeshes(iObject)
            CalculateObjectTangents(iObject,m)
         next m
      endif
      tObject=-1
      for o=Effects[iEffect].ObjectFreeUp to Effects[iEffect].ObjectCount
         if Effects[iEffect].Object[o]=0
            tObject=o:exit
         endif
      next o
      if tObject=-1
         inc Effects[iEffect].ObjectCount
         Effects[iEffect].Object.length=Effects[iEffect].ObjectCount+1
         tObject=Effects[iEffect].ObjectCount
      endif
      Effects[iEffect].ObjectFreeUp=tObject
      if tObject>Effects[iEffect].ObjectTotal then Effects[iEffect].ObjectTotal=tObject
      Effects[iEffect].Object[tObject]=iObject
      tShaderTeq=Effects[iEffect].Technique
      if Effects[iEffect].ShaderCount>-1 then SetObjectShader(iObject,Effects[iEffect].Shader[tShaderTeq])
   endif
endfunction

`--------------
` Remove Object Mesh Effect
`--------------
function RemoveObjectMeshEffect(iObject as integer,iMesh as integer)
   if GetObjectexists(iObject)=1 then SetObjectmeshShader(iObject,iMesh,0)
   for e=0 to EffectsCount
      for o=0 to Effects[e].ObjectCount
         if Effects[e].Object[o]=iObject
            if Effects[e].ObjectFreeUp>o then Effects[e].ObjectFreeUp=o
            Effects[e].Object[o]=0 : inc Effects[e].ObjectRestack
         endif
      next o
      for m=0 to Effects[e].MeshCount
         if Effects[e].MeshObject[m]=iObject and Effects[e].Mesh[m]=iMesh
            if Effects[e].MeshFreeUp>m then Effects[e].MeshFreeUp=m
            Effects[e].Mesh[m]=0 : inc Effects[e].MeshRestack
            Effects[e].MeshObject[m]=0
            exit
         endif
      next m
      EffectRestackObjectArray(e)
      EffectRestackMeshArray(e)
   next e
endfunction
// Restack Effects meshes array to reduce fragmentation
function EffectRestackMeshArray(iEffect as integer)
   if Effects[iEffect].MeshRestack>25
      Effects[iEffect].MeshRestack=0
      tNextEelement=Effects[iEffect].MeshFreeUp
      tCurrEelement=Effects[iEffect].MeshFreeUp
      repeat
         if Effects[iEffect].Mesh[tNextEelement]>0
            Effects[iEffect].Mesh[tCurrEelement]=Effects[iEffect].Mesh[tNextEelement]
            Effects[iEffect].MeshObject[tCurrEelement]=Effects[iEffect].MeshObject[tNextEelement]
            if tNextEelement>tCurrEelement 
               Effects[iEffect].Mesh[tNextEelement]=0
               Effects[iEffect].MeshObject[tNextEelement]=0
            endif
            inc tCurrEelement
         endif
         inc tNextEelement
      until tNextEelement>Effects[iEffect].MeshTotal
      Effects[iEffect].MeshTotal=tCurrEelement
      Effects[iEffect].MeshFreeUp=tCurrEelement
   endif
endfunction

`--------------
` Set Object Mesh Effect
`--------------
function SetObjectMeshEffect(iObject as integer,iMesh as integer,iEffect as integer,iTangents as integer)
   RemoveObjectMeshEffect(iObject,iMesh)
   if GetEffectExist(iEffect)=0 then exitfunction
   if GetObjectexists(iObject)=1
      if iTangents=1 then CalculateObjectTangents(iObject,iMesh)
      tMesh=-1
      for m=Effects[iEffect].MeshFreeUp to Effects[iEffect].MeshCount
         if Effects[iEffect].Mesh[m]=0 
            tMesh=m:exit
         endif
      next m
      if tMesh=-1
         inc Effects[iEffect].MeshCount
         Effects[iEffect].Mesh.length=Effects[iEffect].MeshCount+1
         Effects[iEffect].MeshObject.length=Effects[iEffect].MeshCount+1
         tMesh=Effects[iEffect].MeshCount
      endif
      Effects[iEffect].MeshFreeUp=tMesh
      if tMesh>Effects[iEffect].MeshTotal then Effects[iEffect].MeshTotal=tMesh
      Effects[iEffect].Mesh[tMesh]=iMesh
      Effects[iEffect].MeshObject[tMesh]=iObject
      tShaderTeq=Effects[iEffect].Technique
      if Effects[iEffect].ShaderCount>-1 then SetObjectMeshShader(iObject,iMesh,Effects[iEffect].Shader[tShaderTeq])
   endif
endfunction

`--------------
` Clone Object With an Effect
`--------------
function CloneObjectEffect(iNewObject as integer,iObject as integer)
   CloneObject(iNewObject,iObject)
   for e=0 to EffectsCount
      for o=0 to Effects[e].ObjectCount
         if Effects[e].Object[o]=iObject then SetObjectEffect(iNewObject,e,0)
      next o
      for m=0 to Effects[e].MeshCount
         if Effects[e].MeshObject[m]=iObject then SetObjectMeshEffect(iNewObject,Effects[e].Mesh[m],e,0)
      next m
   next e
endfunction

`--------------
` Set Effect Technique
`--------------
function SetEffectTechnique(iEffect as integer,iTechniqueName as string)
   if GetEffectExist(iEffect)=0 then exitfunction
   if Effects[iEffect].ShaderCount>0
      iTechniqueName=lower(iTechniqueName)
      tOldShaderTeq=Effects[iEffect].Technique
      for t=0 to Effects[iEffect].ShaderCount
         if Effects[iEffect].TechniqueName[t]=iTechniqueName
            Effects[iEffect].Technique=t:exit
         endif
      next t
      tShaderTeq=Effects[iEffect].Technique
      if tOldShaderTeq<>tShaderTeq
         for o=0 to Effects[iEffect].ObjectTotal
            if Effects[iEffect].Object[o]>0
               if GetObjectexists(Effects[iEffect].Object[o])=1
                  SetObjectShader(Effects[iEffect].Object[o],Effects[iEffect].Shader[tShaderTeq])
               else
                  if Effects[iEffect].ObjectFreeUp>o then Effects[iEffect].ObjectFreeUp=o
                  Effects[iEffect].Object[o]=0 : inc Effects[iEffect].ObjectRestack
               endif
            endif
         next o
         for m=0 to Effects[iEffect].MeshTotal
            if Effects[iEffect].MeshObject[m]>0
               if GetObjectexists(Effects[iEffect].MeshObject[m])=1
                  SetObjectmeshShader(Effects[iEffect].MeshObject[m],Effects[iEffect].Mesh[m],Effects[iEffect].Shader[tShaderTeq])
               else
                  if Effects[iEffect].MeshFreeUp>m then Effects[iEffect].MeshFreeUp=m
                  Effects[iEffect].Mesh[m]=0 : inc Effects[iEffect].MeshRestack
                  Effects[iEffect].MeshObject[m]=0
               endif
            endif
         next m
      endif
   endif
endfunction

`--------------
` Set Effect Constant Float
`--------------
function SetEffectConstantFloat(iEffect as integer,iConstant as String,iValue as float)
   if GetEffectExist(iEffect)=0 then exitfunction
   for s=0 to Effects[iEffect].ShaderCount:SetShaderConstantByName(Effects[iEffect].Shader[s],iConstant,iValue,0,0,0):next s
endfunction
function SetEffectConstantFloatElement(iEffect as integer,iConstant as String,iArrayIndex as integer,iValue as float)
   if GetEffectExist(iEffect)=0 then exitfunction
   for s=0 to Effects[iEffect].ShaderCount:SetShaderConstantArrayFloatByName(Effects[iEffect].Shader[s],iConstant,iArrayIndex,iValue):next s
endfunction

`--------------
` Set Effect Constant Vector
`--------------
function SetEffectConstantVector(iEffect as integer,iConstant as String,iVector4 as integer)
   if GetEffectExist(iEffect)=0 then exitfunction
   for s=0 to Effects[iEffect].ShaderCount:SetShaderConstantByName(Effects[iEffect].Shader[s],iConstant,Vector4Float[iVector4].x,Vector4Float[iVector4].y,Vector4Float[iVector4].z,Vector4Float[iVector4].w):next s
endfunction
function SetEffectConstantVectorElement(iEffect as integer,iConstant as String,iArrayIndex as integer,iVector4 as integer)
   if GetEffectExist(iEffect)=0 then exitfunction
   for s=0 to Effects[iEffect].ShaderCount:SetShaderConstantArrayByName(Effects[iEffect].Shader[s],iConstant,iArrayIndex,Vector4Float[iVector4].x,Vector4Float[iVector4].y,Vector4Float[iVector4].z,Vector4Float[iVector4].w):next s
endfunction
function SetEffectConstantVectorElement3(iEffect as integer,iConstant as String,iArrayIndex as integer,iVector3 as integer)
   if GetEffectExist(iEffect)=0 then exitfunction
   for s=0 to Effects[iEffect].ShaderCount:SetShaderConstantArrayVec3ByName(Effects[iEffect].Shader[s],iConstant,iArrayIndex,Vector3Float[iVector3].x,Vector3Float[iVector3].y,Vector3Float[iVector3].z):next s
endfunction
function SetEffectConstantVectorElement2(iEffect as integer,iConstant as String,iArrayIndex as integer,iVector2 as integer)
   if GetEffectExist(iEffect)=0 then exitfunction
   for s=0 to Effects[iEffect].ShaderCount:SetShaderConstantArrayVec2ByName(Effects[iEffect].Shader[s],iConstant,iArrayIndex,Vector2Float[iVector2].x,Vector2Float[iVector2].y):next s
endfunction

`--------------
` Set Effect Constant Matrix
`--------------
function SetEffectConstantMatrix(iEffect as integer,iConstant as String,iMatrix as integer)
   if GetEffectExist(iEffect)=0 then exitfunction
   for s=0 to Effects[iEffect].ShaderCount
      SetShaderConstantArrayVec4ByName(Effects[iEffect].Shader[s],iConstant,0,Matrix4Float[iMatrix,1].x,Matrix4Float[iMatrix,1].y,Matrix4Float[iMatrix,1].z,Matrix4Float[iMatrix,1].w)
      SetShaderConstantArrayVec4ByName(Effects[iEffect].Shader[s],iConstant,1,Matrix4Float[iMatrix,2].x,Matrix4Float[iMatrix,2].y,Matrix4Float[iMatrix,2].z,Matrix4Float[iMatrix,2].w)
      SetShaderConstantArrayVec4ByName(Effects[iEffect].Shader[s],iConstant,2,Matrix4Float[iMatrix,3].x,Matrix4Float[iMatrix,3].y,Matrix4Float[iMatrix,3].z,Matrix4Float[iMatrix,3].w)
      SetShaderConstantArrayVec4ByName(Effects[iEffect].Shader[s],iConstant,3,Matrix4Float[iMatrix,4].x,Matrix4Float[iMatrix,4].y,Matrix4Float[iMatrix,4].z,Matrix4Float[iMatrix,4].w)
   next s
endfunction
function SetEffectConstantMatrix3(iEffect as integer,iConstant as String,iMatrix as integer)
   if GetEffectExist(iEffect)=0 then exitfunction
   for s=0 to Effects[iEffect].ShaderCount
      SetShaderConstantArrayVec3ByName(Effects[iEffect].Shader[s],iConstant,0,Matrix4Float[iMatrix,1].x,Matrix4Float[iMatrix,1].y,Matrix4Float[iMatrix,1].z)
      SetShaderConstantArrayVec3ByName(Effects[iEffect].Shader[s],iConstant,1,Matrix4Float[iMatrix,2].x,Matrix4Float[iMatrix,2].y,Matrix4Float[iMatrix,2].z)
      SetShaderConstantArrayVec3ByName(Effects[iEffect].Shader[s],iConstant,2,Matrix4Float[iMatrix,3].x,Matrix4Float[iMatrix,3].y,Matrix4Float[iMatrix,3].z)
   next s
endfunction
function SetEffectConstantMatrix2(iEffect as integer,iConstant as String,iMatrix as integer)
   if GetEffectExist(iEffect)=0 then exitfunction
   for s=0 to Effects[iEffect].ShaderCount 
      SetShaderConstantArrayVec2ByName(Effects[iEffect].Shader[s],iConstant,0,Matrix4Float[iMatrix,1].x,Matrix4Float[iMatrix,1].y)
      SetShaderConstantArrayVec2ByName(Effects[iEffect].Shader[s],iConstant,1,Matrix4Float[iMatrix,2].x,Matrix4Float[iMatrix,2].y)
   next s
endfunction

`--------------
` Set View Size Vector 
`--------------
function SetViewVec(iVector4 as integer,iViewX as float,iViewY as float)
   set_vector4(iVector4,0.5/iViewX,0.5/iViewY,1,1)
endfunction iVector4

`--------------
` Calculate Object Normals
`--------------
function CalculateObjectNormals(iObject as integer)
   for m=1 to GetObjectNumMeshes(iObject)
      tMemblock=CreateMemblockFromObjectMesh(iObject,m)
      tVertexCount=GetMemblockInt(tMemblock,0)
      tIndexCount=GetMemblockInt(tMemblock,4)
      local dim tVertexNormals[tVertexCount] as Vec3Float
      if tIndexCount=0 then tIndices=tVertexCount else tIndices=tIndexCount
      tPos=GetMemblockInt(tMemblock,20)
      for v=0 to tIndices-1 step 3
         if tIndexCount>0
            i1=GetMemblockInt(tMemblock,tPos):inc tPos,4
            i2=GetMemblockInt(tMemblock,tPos):inc tPos,4
            i3=GetMemblockInt(tMemblock,tPos):inc tPos,4   
         else
            i1=v+0 : i2=v+1 : i3=v+2
         endif
         Vector4Float[262].x=GetMeshMemblockVertexX(tMemblock,i2)-GetMeshMemblockVertexX(tMemblock,i1)
         Vector4Float[263].x=GetMeshMemblockVertexX(tMemblock,i3)-GetMeshMemblockVertexX(tMemblock,i1)
         Vector4Float[262].y=GetMeshMemblockVertexY(tMemblock,i2)-GetMeshMemblockVertexY(tMemblock,i1)
         Vector4Float[263].y=GetMeshMemblockVertexY(tMemblock,i3)-GetMeshMemblockVertexY(tMemblock,i1)
         Vector4Float[262].z=GetMeshMemblockVertexZ(tMemblock,i2)-GetMeshMemblockVertexZ(tMemblock,i1)
         Vector4Float[263].z=GetMeshMemblockVertexZ(tMemblock,i3)-GetMeshMemblockVertexZ(tMemblock,i1)
         SetVector3(Vector3AGK.x,Vector4Float[262].x,Vector4Float[262].y,Vector4Float[262].z)
         SetVector3(Vector3AGK.y,Vector4Float[263].x,Vector4Float[263].y,Vector4Float[263].z)
         GetVector3Cross(Vector3AGK.z,Vector3AGK.x,Vector3AGK.y)
         Vector4Float[261].w=GetVector3Length(Vector3AGK.z)
         SetVector3(Vector3AGK.z,getVector3X(Vector3AGK.z)/Vector4Float[261].w,getVector3Y(Vector3AGK.z)/Vector4Float[261].w,getVector3Z(Vector3AGK.z)/Vector4Float[261].w)
         tVertexNormals[i1].x=tVertexNormals[i1].x+getVector3X(Vector3AGK.z)
         tVertexNormals[i1].y=tVertexNormals[i1].y+getVector3Y(Vector3AGK.z)
         tVertexNormals[i1].z=tVertexNormals[i1].z+getVector3Z(Vector3AGK.z)
         tVertexNormals[i2].x=tVertexNormals[i2].x+getVector3X(Vector3AGK.z)
         tVertexNormals[i2].y=tVertexNormals[i2].y+getVector3Y(Vector3AGK.z)
         tVertexNormals[i2].z=tVertexNormals[i2].z+getVector3Z(Vector3AGK.z)
         tVertexNormals[i3].x=tVertexNormals[i3].x+getVector3X(Vector3AGK.z)
         tVertexNormals[i3].y=tVertexNormals[i3].y+getVector3Y(Vector3AGK.z)
         tVertexNormals[i3].z=tVertexNormals[i3].z+getVector3Z(Vector3AGK.z)
      next v
      for v=0 to tVertexCount-1
         SetVector3(Vector3AGK.x,tVertexNormals[v].x,tVertexNormals[v].y,tVertexNormals[v].z)
         Vector4Float[261].w=GetVector3Length(Vector3AGK.x)
         SetMeshMemblockVertexNormal(tMemblock,v,tVertexNormals[v].x/Vector4Float[261].w,tVertexNormals[v].y/Vector4Float[261].w,tVertexNormals[v].z/Vector4Float[261].w)
      next v
      undim tVertexNormals[]
      SetObjectMeshFromMemblock(iObject,m,tMemblock)
      DeleteMemblock(tMemblock)
   next m
endfunction

`--------------
` Calculate Object Tangents
`--------------
function CalculateObjectTangents(iObject as integer,iMeshIndex as integer)
   tOldMemblock=CreateMemblockFromObjectMesh(iObject,iMeshIndex)
   for v=20 to GetMemblockInt(tOldMemblock,16) step 4
     if GetMemblockString(tOldMemblock,v,7)="tangent" then exitfunction
   next v
   tOldMemblockSize=GetMemblockSize(tOldMemblock)
   tOldAttributes=GetMemblockInt(tOldMemblock,8)
   tOldVertexSize=GetMemblockInt(tOldMemblock,12)
   tOldVertexData=GetMemblockInt(tOldMemblock,16)
   tOldIndexData=GetMemblockInt(tOldMemblock,20)
   tNewVertexCount=GetMemblockInt(tOldMemblock,0)
   tNewIndexCount=GetMemblockInt(tOldMemblock,4)
   tNewAttributes=tOldAttributes+1
   tNewVertexSize=tOldVertexSize+12
   tNewVertexData=tOldVertexData+12
   tNewIndexData=tNewVertexData+(tNewVertexCount*tNewVertexSize)
   tNewMemblock=CreateMemblock(tNewIndexData+(tNewIndexCount*4))
   SetMemblockInt(tNewMemblock,0,tNewVertexCount)
   SetMemblockInt(tNewMemblock,4,tNewIndexCount)
   SetMemblockint(tNewMemblock,8,tNewAttributes)
   Setmemblockint(tNewMemblock,12,tNewVertexSize)
   SetMemblockInt(tNewMemblock,16,tNewVertexData)
   SetMemblockInt(tNewMemblock,20,tNewIndexData)
   tSize=tOldVertexData-24
   CopyMemblock(tOldMemblock,tNewMemblock,24,24,tSize)
   SetMemblockInt(tNewMemblock,tOldVertexData,0x08000300)
   SetMemblockString(tNewMemblock,tOldVertexData+4,"tangent")
   if tNewIndexCount>0
      tSize=tNewIndexCount*4
      CopyMemblock(tOldMemblock,tNewMemblock,tOldIndexData,tNewIndexData,tSize)
   endif
   local dim tVertexTangent[tNewVertexCount] as Vec3Float
   if tNewIndexCount=0 then tIndices=tNewVertexCount else tIndices=tNewIndexCount
   tPos=tOldIndexData
   for v=0 to tIndices-1 step 3
      if tNewIndexCount>0
         i1=GetMemblockInt(tOldMemblock,tPos):inc tPos,4
         i2=GetMemblockInt(tOldMemblock,tPos):inc tPos,4
         i3=GetMemblockInt(tOldMemblock,tPos):inc tPos,4   
      else
         i1=v+0 : i2=v+1 : i3=v+2
      endif
      Vector4Float[262].x=GetMeshMemblockVertexX(tOldMemblock,i2)-GetMeshMemblockVertexX(tOldMemblock,i1)
      Vector4Float[263].x=GetMeshMemblockVertexX(tOldMemblock,i3)-GetMeshMemblockVertexX(tOldMemblock,i1)
      Vector4Float[262].y=GetMeshMemblockVertexY(tOldMemblock,i2)-GetMeshMemblockVertexY(tOldMemblock,i1)
      Vector4Float[263].y=GetMeshMemblockVertexY(tOldMemblock,i3)-GetMeshMemblockVertexY(tOldMemblock,i1)
      Vector4Float[262].z=GetMeshMemblockVertexZ(tOldMemblock,i2)-GetMeshMemblockVertexZ(tOldMemblock,i1)
      Vector4Float[263].z=GetMeshMemblockVertexZ(tOldMemblock,i3)-GetMeshMemblockVertexZ(tOldMemblock,i1)
      Vector2Float[262].x=GetMeshMemblockVertexu(tOldMemblock,i2)-GetMeshMemblockVertexu(tOldMemblock,i1)
      Vector2Float[263].x=GetMeshMemblockVertexu(tOldMemblock,i3)-GetMeshMemblockVertexu(tOldMemblock,i1)
      Vector2Float[262].y=GetMeshMemblockVertexv(tOldMemblock,i2)-GetMeshMemblockVertexv(tOldMemblock,i1)
      Vector2Float[263].y=GetMeshMemblockVertexv(tOldMemblock,i3)-GetMeshMemblockVertexv(tOldMemblock,i1)
      SetVector3(Vector3AGK.x,Vector4Float[262].x,Vector4Float[262].y,Vector4Float[262].z)
      SetVector3(Vector3AGK.y,Vector4Float[263].x,Vector4Float[263].y,Vector4Float[263].z)
      GetVector3Cross(Vector3AGK.z,Vector3AGK.x,Vector3AGK.y)
      Vector4Float[261].w=GetVector3Length(Vector3AGK.z)
      SetVector3(Vector3AGK.z,getVector3X(Vector3AGK.z)/Vector4Float[261].w,getVector3Y(Vector3AGK.z)/Vector4Float[261].w,getVector3Z(Vector3AGK.z)/Vector4Float[261].w)
      Vector4Float[261].w=1.0/(Vector2Float[262].x*Vector2Float[263].y-Vector2Float[263].x*Vector2Float[262].y)
      Vector4Float[261].x=(Vector2Float[263].y*Vector4Float[262].x-Vector2Float[262].y*Vector4Float[263].x)*Vector4Float[261].w
      Vector4Float[261].y=(Vector2Float[263].y*Vector4Float[262].y-Vector2Float[262].y*Vector4Float[263].y)*Vector4Float[261].w
      Vector4Float[261].z=(Vector2Float[263].y*Vector4Float[262].z-Vector2Float[262].y*Vector4Float[263].z)*Vector4Float[261].w
      SetVector3(Vector3AGK.y,Vector4Float[261].x,Vector4Float[261].y,Vector4Float[261].z)
      Vector4Float[261].w=GetVector3Dot(Vector3AGK.z,Vector3AGK.y)
      Vector4Float[261].x=Vector4Float[261].x-(GetVector3X(Vector3AGK.z)*Vector4Float[261].w)
      Vector4Float[261].y=Vector4Float[261].y-(GetVector3Y(Vector3AGK.z)*Vector4Float[261].w)
      Vector4Float[261].z=Vector4Float[261].z-(GetVector3Z(Vector3AGK.z)*Vector4Float[261].w)
      tVertexTangent[i1].x=tVertexTangent[i1].x+Vector4Float[261].x
      tVertexTangent[i1].y=tVertexTangent[i1].y+Vector4Float[261].y
      tVertexTangent[i1].z=tVertexTangent[i1].z+Vector4Float[261].z
      tVertexTangent[i2].x=tVertexTangent[i2].x+Vector4Float[261].x
      tVertexTangent[i2].y=tVertexTangent[i2].y+Vector4Float[261].y
      tVertexTangent[i2].z=tVertexTangent[i2].z+Vector4Float[261].z
      tVertexTangent[i3].x=tVertexTangent[i3].x+Vector4Float[261].x
      tVertexTangent[i3].y=tVertexTangent[i3].y+Vector4Float[261].y
      tVertexTangent[i3].z=tVertexTangent[i3].z+Vector4Float[261].z
   next v
   tOldVPos=tOldVertexData
   tNewVPos=tNewVertexData
   for v=0 to tNewVertexCount-1
      CopyMemblock(tOldMemblock,tNewMemblock,tOldVPos,tNewVPos,tOldVertexSize)
      inc tOldVPos,tOldVertexSize
      inc tNewVPos,tOldVertexSize
      SetVector3(Vector3AGK.x,tVertexTangent[v].x,tVertexTangent[v].y,tVertexTangent[v].z)
      Vector4Float[261].w=GetVector3Length(Vector3AGK.x)
      setMemblockFloat(tNewMemblock,tNewVPos,tVertexTangent[v].x/Vector4Float[261].w)
      SetMemblockFloat(tNewMemblock,tNewVPos+4,tVertexTangent[v].y/Vector4Float[261].w)
      SetMemblockFloat(tNewMemblock,tNewVPos+8,tVertexTangent[v].z/Vector4Float[261].w)
      inc tNewVPos,12
   next v
   DeleteMemblock(tOldMemblock)
   SetObjectMeshFromMemblock(iObject,iMeshIndex,tNewMemblock)
   DeleteMemblock(tNewMemblock)
   undim tVertexTangent[]
endfunction

`--------------
` Get Object Axis Aligned Bounding Box
`--------------
function AxisAlignedBoundingBox(iObject as integer)
   RotateX_Matrix4(1,WrapValue(GetObjectAngleX(iObject))*0.01745329251)
   RotateY_Matrix4(2,WrapValue(GetObjectAngleY(iObject))*0.01745329251)
   RotateZ_Matrix4(3,WrapValue(GetObjectAngleZ(iObject))*0.01745329251)
   Multiply_Matrix4(4,1,2)
   Multiply_Matrix4(4,4,3)
   Vector4F.x=(GetObjectSizeMaxX(iObject)-GetObjectSizeMinX(iObject))*0.5
   Vector4F.y=(GetObjectSizeMaxY(iObject)-GetObjectSizeMinY(iObject))*0.5
   Vector4F.z=(GetObjectSizeMaxZ(iObject)-GetObjectSizeMinZ(iObject))*0.5
   MinMaxVecReset()
   for b=0 to 7
      Set_Vector3(1,Vector4F.x*AABBox[b].x,Vector4F.y*AABBox[b].y,Vector4F.z*AABBox[b].z)
      Transform_Coords_Vector3(1,1,4)
      MinMaxVec(Vector3Float[1].x,Vector3Float[1].y,Vector3Float[1].z)
   next b
   MinMaxPositionSize()
   MinMaxPos.x=GetObjectWorldX(iObject)
   MinMaxPos.y=GetObjectWorldY(iObject)
   MinMaxPos.z=GetObjectWorldZ(iObject)
endfunction

`--------------
` Min Max Reset
`--------------
function MinMaxVecReset()
   MinVec.x=9999999999:MinVec.y=9999999999:MinVec.z=9999999999
   MaxVec.x=-9999999999:MaxVec.y=-9999999999:MaxVec.z=-9999999999
endfunction

`--------------
` Min Max
`--------------
function MinMaxVec(iX as float,iY as float,iZ as float)
   if iX<MinVec.x then MinVec.x=iX
   if iX>MaxVec.x then MaxVec.x=iX
   if iY<MinVec.y then MinVec.y=iY
   if iY>MaxVec.y then MaxVec.y=iY
   if iZ<MinVec.z then MinVec.z=iZ
   if iZ>MaxVec.z then MaxVec.z=iZ
endfunction

`--------------
` Min Max Position/Size
`--------------
function MinMaxPositionSize()
   MinMaxSize.x=(MaxVec.x-MinVec.x)
   MinMaxSize.y=(MaxVec.y-MinVec.y)
   MinMaxSize.z=(MaxVec.z-MinVec.z)
   MinMaxPos.x=MinVec.x+(MinMaxSize.x/2)
   MinMaxPos.y=MinVec.y+(MinMaxSize.y/2)
   MinMaxPos.z=MinVec.z+(MinMaxSize.z/2)
endfunction

`--------------
` Box Collision
`--------------
function BoxCollision(iX as float,iY as float,iZ as float,ipX as float,ipY as float,ipZ as float,iSx as float,iSy as float,iSz as float)
   if iX<(ipX+(iSx/2)) and iX>(ipX-(iSx/2))
      if iY<(ipY+(iSy/2)) and iY>(ipY-(iSy/2))
         if iZ<(ipZ+(iSz/2)) and iZ>(ipZ-(iSz/2)) then exitfunction 1
      endif
   endif
endfunction 0

`--------------
` In Normal
`--------------
function InNormal(iFromX as float,iFromY as float,iFromZ as float,iX as float,iY as float,iZ as float,iNorX as float,iNorY as float,iNorZ as float)
   iFromX=iFromX-iX : iFromY=iFromY-iY : iFromZ=iFromZ-iZ
   r#=iNorX*iFromX+iNorY*iFromY+iNorZ*iFromZ
endfunction r#

`--------------
` Plane Ray Cast
`--------------
function RayPlaneCast(iFromX as float,iFromY as float,iFromZ as float,iToX as float,iToY as float,iToZ as float,iX as float,iY as float,iZ as float,iNorX as float,iNorY as float,iNorZ as float)
   iFromX=iFromX-iX : iFromY=iFromY-iY : iFromZ=iFromZ-iZ
   iToX=iToX-iX : iToY=iToY-iY : iToZ=iToZ-iZ
   Vector4Float[0].x=iNorX*(iToX-iFromX)+iNorY*(iToY-iFromY)+iNorZ*(iToZ-iFromZ)
   Vector4Float[0].w=-(iNorX*iFromX+iNorY*iFromY+iNorZ*iFromZ)/Vector4Float[0].x
   Vector4Float[0].x=iX+(iFromX+Vector4Float[0].w*(iToX-iFromX))
   Vector4Float[0].y=iY+(iFromY+Vector4Float[0].w*(iToY-iFromY))
   Vector4Float[0].z=iZ+(iFromZ+Vector4Float[0].w*(iToZ-iFromZ))
endfunction

`--------------
` Box Ray Cast
`--------------
function RayBoxCast(iFromX as float,iFromY as float,iFromZ as float,iToX as float,iToY as float,iToZ as float,iMinX as float,iMaxX as float,iMinY as float,iMaxY as float,iMinZ as float,iMaxZ as float)
   RayPlaneCast(iFromX,iFromY,iFromZ,iToX,iToY,iToZ,iMinX,0,0,-1,0,0)
   if Vector4Float[0].y>iMinY and Vector4Float[0].y<iMaxY and Vector4Float[0].z>iMinZ and Vector4Float[0].z<iMaxZ then exitfunction 1
   RayPlaneCast(iFromX,iFromY,iFromZ,iToX,iToY,iToZ,iMaxX,0,0,1,0,0)
   if Vector4Float[0].y>iMinY and Vector4Float[0].y<iMaxY and Vector4Float[0].z>iMinZ and Vector4Float[0].z<iMaxZ then exitfunction 1
   RayPlaneCast(iFromX,iFromY,iFromZ,iToX,iToY,iToZ,0,iMinY,0,0,-1,0)
   if Vector4Float[0].x>iMinX and Vector4Float[0].x<iMaxX and Vector4Float[0].z>iMinZ and Vector4Float[0].z<iMaxZ then exitfunction 1
   RayPlaneCast(iFromX,iFromY,iFromZ,iToX,iToY,iToZ,0,iMaxY,0,0,1,0)
   if Vector4Float[0].x>iMinX and Vector4Float[0].x<iMaxX and Vector4Float[0].z>iMinZ and Vector4Float[0].z<iMaxZ then exitfunction 1
   RayPlaneCast(iFromX,iFromY,iFromZ,iToX,iToY,iToZ,0,0,iMinZ,0,0,-1)
   if Vector4Float[0].x>iMinX and Vector4Float[0].x<iMaxX and Vector4Float[0].y>iMinY and Vector4Float[0].y<iMaxY then exitfunction 1
   RayPlaneCast(iFromX,iFromY,iFromZ,iToX,iToY,iToZ,0,0,iMaxZ,0,0,1)
   if Vector4Float[0].x>iMinX and Vector4Float[0].x<iMaxX and Vector4Float[0].y>iMinY and Vector4Float[0].y<iMaxY then exitfunction 1
endfunction 0

`--------------
` Triangle Ray Cast
`--------------
function RayTriangleCast(iFromX as float,iFromY as float,iFromZ as float,iToX as float,iToY as float,iToZ as float)
   iToX=iToX-iFromX : iToY=iToY-iFromY : iToZ=iToZ-iFromZ
   Vector4Float[261].x=Vector4Float[3].x-Vector4Float[1].x
   Vector4Float[261].y=Vector4Float[3].y-Vector4Float[1].y
   Vector4Float[261].z=Vector4Float[3].z-Vector4Float[1].z
   Vector4Float[262].x=Vector4Float[2].x-Vector4Float[1].x
   Vector4Float[262].y=Vector4Float[2].y-Vector4Float[1].y
   Vector4Float[262].z=Vector4Float[2].z-Vector4Float[1].z
   Vector4Float[263].x=(iToY*Vector4Float[262].z)-(Vector4Float[262].y*iToZ)
   Vector4Float[263].y=(iToZ*Vector4Float[262].x)-(Vector4Float[262].z*iToX)
   Vector4Float[263].z=(iToX*Vector4Float[262].y)-(Vector4Float[262].x*iToY)
   Vector4Float[261].w=(Vector4Float[261].x* Vector4Float[263].x)+(Vector4Float[261].y*Vector4Float[263].y)+(Vector4Float[261].z*Vector4Float[263].z)
   if Vector4Float[261].w>-0.00001 and Vector4Float[261].w<0.00001 then exitfunction 0
   Vector4Float[261].w=1.0/Vector4Float[261].w
   Vector4Float[264].x=iFromX-Vector4Float[1].x
   Vector4Float[264].y=iFromY-Vector4Float[1].y
   Vector4Float[264].z=iFromZ-Vector4Float[1].z
   Vector4Float[262].w=Vector4Float[261].w*((Vector4Float[264].x*Vector4Float[263].x)+(Vector4Float[264].y*Vector4Float[263].y)+(Vector4Float[264].z*Vector4Float[263].z))
   if Vector4Float[262].w<0 or Vector4Float[262].w>1 then exitfunction 0
   Vector4Float[265].x=(Vector4Float[264].y*Vector4Float[261].z)-(Vector4Float[261].y*Vector4Float[264].z)
   Vector4Float[265].y=(Vector4Float[264].z*Vector4Float[261].x)-(Vector4Float[261].z*Vector4Float[264].x)
   Vector4Float[265].z=(Vector4Float[264].x*Vector4Float[261].y)-(Vector4Float[261].x*Vector4Float[264].y)
   Vector4Float[263].w=Vector4Float[261].w*((iToX*Vector4Float[265].x)+(iToY*Vector4Float[265].y)+(iToZ*Vector4Float[265].z))
   if Vector4Float[263].w<0.0 or Vector4Float[262].w+Vector4Float[263].w>1.0 then exitfunction 0
   Vector4Float[264].w=Vector4Float[261].w*((Vector4Float[262].x*Vector4Float[265].x)+(Vector4Float[262].y*Vector4Float[265].y)+(Vector4Float[262].z*Vector4Float[265].z))
   if Vector4Float[264].w<0 or Vector4Float[264].w>1 then exitfunction 0
   Vector4Float[266].x=(Vector4Float[261].y*Vector4Float[262].z)-(Vector4Float[261].z*Vector4Float[262].y)
   Vector4Float[266].y=(Vector4Float[261].z*Vector4Float[262].x)-(Vector4Float[261].x*Vector4Float[262].z)
   Vector4Float[266].z=(Vector4Float[261].x*Vector4Float[262].y)-(Vector4Float[261].y*Vector4Float[262].x)
   Vector4Float[267].x=-Vector4Float[266].x*Vector4Float[1].x-Vector4Float[266].y*Vector4Float[1].y-Vector4Float[266].z*Vector4Float[1].z
   Vector4Float[267].y=Vector4Float[266].x*iToX+Vector4Float[266].y*iToY+Vector4Float[266].z*iToZ
   Vector4Float[267].z=-(Vector4Float[267].x+Vector4Float[266].x*iFromX+Vector4Float[266].y*iFromY+Vector4Float[266].z*iFromZ)/Vector4Float[267].y
   Vector4Float[0].x=iFromX+Vector4Float[267].z*iToX
   Vector4Float[0].y=iFromY+Vector4Float[267].z*iToY
   Vector4Float[0].z=iFromZ+Vector4Float[267].z*iToZ
endfunction 1

`--------------
` Get View Project Frustum 
`--------------
function GetViewProjectFrustum(iFrustumIndex as integer,iMatrix as integer)
   tPos=0
   for i=1 to 4
      ViewProjectVec[tPos+0]=Matrix4Float[iMatrix,i].x
      ViewProjectVec[tPos+1]=Matrix4Float[iMatrix,i].y
      ViewProjectVec[tPos+2]=Matrix4Float[iMatrix,i].z
      ViewProjectVec[tPos+3]=Matrix4Float[iMatrix,i].w
      inc tPos,4
   next i
   ViewProjectFrustumVec[iFrustumIndex,0,0]=ViewProjectVec[3]-ViewProjectVec[0]
   ViewProjectFrustumVec[iFrustumIndex,0,1]=ViewProjectVec[7]-ViewProjectVec[4]
   ViewProjectFrustumVec[iFrustumIndex,0,2]=ViewProjectVec[11]-ViewProjectVec[8]
   ViewProjectFrustumVec[iFrustumIndex,0,3]=ViewProjectVec[15]-ViewProjectVec[12]
   ViewProjectFrustumVec[iFrustumIndex,1,0]=ViewProjectVec[3]+ViewProjectVec[0]
   ViewProjectFrustumVec[iFrustumIndex,1,1]=ViewProjectVec[7]+ViewProjectVec[4]
   ViewProjectFrustumVec[iFrustumIndex,1,2]=ViewProjectVec[11]+ViewProjectVec[8]
   ViewProjectFrustumVec[iFrustumIndex,1,3]=ViewProjectVec[15]+ViewProjectVec[12]
   ViewProjectFrustumVec[iFrustumIndex,2,0]=ViewProjectVec[3]+ViewProjectVec[1]
   ViewProjectFrustumVec[iFrustumIndex,2,1]=ViewProjectVec[7]+ViewProjectVec[5]
   ViewProjectFrustumVec[iFrustumIndex,2,2]=ViewProjectVec[11]+ViewProjectVec[9]
   ViewProjectFrustumVec[iFrustumIndex,2,3]=ViewProjectVec[15]+ViewProjectVec[13]
   ViewProjectFrustumVec[iFrustumIndex,3,0]=ViewProjectVec[3]-ViewProjectVec[1]
   ViewProjectFrustumVec[iFrustumIndex,3,1]=ViewProjectVec[7]-ViewProjectVec[5]
   ViewProjectFrustumVec[iFrustumIndex,3,2]=ViewProjectVec[11]-ViewProjectVec[9]
   ViewProjectFrustumVec[iFrustumIndex,3,3]=ViewProjectVec[15]-ViewProjectVec[13]
   ViewProjectFrustumVec[iFrustumIndex,4,0]=ViewProjectVec[3]-ViewProjectVec[2]
   ViewProjectFrustumVec[iFrustumIndex,4,1]=ViewProjectVec[7]-ViewProjectVec[6]
   ViewProjectFrustumVec[iFrustumIndex,4,2]=ViewProjectVec[11]-ViewProjectVec[10]
   ViewProjectFrustumVec[iFrustumIndex,4,3]=ViewProjectVec[15]-ViewProjectVec[14]
   ViewProjectFrustumVec[iFrustumIndex,5,0]=ViewProjectVec[3]+ViewProjectVec[2]
   ViewProjectFrustumVec[iFrustumIndex,5,1]=ViewProjectVec[7]+ViewProjectVec[6]
   ViewProjectFrustumVec[iFrustumIndex,5,2]=ViewProjectVec[11]+ViewProjectVec[10]
   ViewProjectFrustumVec[iFrustumIndex,5,3]=ViewProjectVec[15]+ViewProjectVec[14]
   SetVector3(Vector3AGK.x,ViewProjectFrustumVec[iFrustumIndex,0,0],ViewProjectFrustumVec[iFrustumIndex,0,1],ViewProjectFrustumVec[iFrustumIndex,0,2])
   Vector4F.x=1.0/getvector3length(Vector3AGK.x)
   SetVector3(Vector3AGK.x,ViewProjectFrustumVec[iFrustumIndex,2,0],ViewProjectFrustumVec[iFrustumIndex,2,1],ViewProjectFrustumVec[iFrustumIndex,2,2])
   Vector4F.y=1.0/getvector3length(Vector3AGK.x)
   SetVector3(Vector3AGK.x,ViewProjectFrustumVec[iFrustumIndex,4,0],ViewProjectFrustumVec[iFrustumIndex,4,1],ViewProjectFrustumVec[iFrustumIndex,4,2])
   Vector4F.z=1.0/getvector3length(Vector3AGK.x)
   SetVector3(Vector3AGK.x,ViewProjectFrustumVec[iFrustumIndex,5,0],ViewProjectFrustumVec[iFrustumIndex,5,1],ViewProjectFrustumVec[iFrustumIndex,5,2])
   Vector4F.w=1.0/getvector3length(Vector3AGK.x)
   for c=0 to 3
      ViewProjectFrustumVec[iFrustumIndex,0,c]=ViewProjectFrustumVec[iFrustumIndex,0,c]*Vector4F.x
      ViewProjectFrustumVec[iFrustumIndex,1,c]=ViewProjectFrustumVec[iFrustumIndex,1,c]*Vector4F.x
      ViewProjectFrustumVec[iFrustumIndex,2,c]=ViewProjectFrustumVec[iFrustumIndex,2,c]*Vector4F.y
      ViewProjectFrustumVec[iFrustumIndex,3,c]=ViewProjectFrustumVec[iFrustumIndex,3,c]*Vector4F.y
      ViewProjectFrustumVec[iFrustumIndex,4,c]=ViewProjectFrustumVec[iFrustumIndex,4,c]*Vector4F.z
      ViewProjectFrustumVec[iFrustumIndex,5,c]=ViewProjectFrustumVec[iFrustumIndex,5,c]*Vector4F.w
   next c
endfunction

`--------------
` Sphere In Frustum
`--------------
function GetSphereInFrustum(iFrustumIndex as integer,iPosX as float,iPosY as float,iPosZ as float,iRadius as float)
   if (ViewProjectFrustumVec[iFrustumIndex,4,0]*iPosX)+(ViewProjectFrustumVec[iFrustumIndex,4,1]*iPosY)+(ViewProjectFrustumVec[iFrustumIndex,4,2]*iPosZ)+ViewProjectFrustumVec[iFrustumIndex,4,3]<-iRadius then exitfunction 0
   if (ViewProjectFrustumVec[iFrustumIndex,0,0]*iPosX)+(ViewProjectFrustumVec[iFrustumIndex,0,1]*iPosY)+(ViewProjectFrustumVec[iFrustumIndex,0,2]*iPosZ)+ViewProjectFrustumVec[iFrustumIndex,0,3]<-iRadius then exitfunction 0
   if (ViewProjectFrustumVec[iFrustumIndex,1,0]*iPosX)+(ViewProjectFrustumVec[iFrustumIndex,1,1]*iPosY)+(ViewProjectFrustumVec[iFrustumIndex,1,2]*iPosZ)+ViewProjectFrustumVec[iFrustumIndex,1,3]<-iRadius then exitfunction 0
   if (ViewProjectFrustumVec[iFrustumIndex,2,0]*iPosX)+(ViewProjectFrustumVec[iFrustumIndex,2,1]*iPosY)+(ViewProjectFrustumVec[iFrustumIndex,2,2]*iPosZ)+ViewProjectFrustumVec[iFrustumIndex,2,3]<-iRadius then exitfunction 0
   if (ViewProjectFrustumVec[iFrustumIndex,3,0]*iPosX)+(ViewProjectFrustumVec[iFrustumIndex,3,1]*iPosY)+(ViewProjectFrustumVec[iFrustumIndex,3,2]*iPosZ)+ViewProjectFrustumVec[iFrustumIndex,3,3]<-iRadius then exitfunction 0
endfunction 1

`--------------
` Box In Frustum
`--------------
function GetBoxInFrustum(iFrustumIndex as integer,iPosX as float,iPosY as float,iPosZ as float,iSizeX as float,iSizeY as float,iSizeZ as float)
   for e=0 to 5
      Vector4Float[1].x=ViewProjectFrustumVec[iFrustumIndex,e,0]*(iPosX-iSizeX)
      Vector4Float[2].x=ViewProjectFrustumVec[iFrustumIndex,e,0]*(iPosX+iSizeX)
      Vector4Float[1].y=ViewProjectFrustumVec[iFrustumIndex,e,1]*(iPosY-iSizeY)
      Vector4Float[2].y=ViewProjectFrustumVec[iFrustumIndex,e,1]*(iPosY+iSizeY)
      Vector4Float[1].z=ViewProjectFrustumVec[iFrustumIndex,e,2]*(iPosZ-iSizeZ)
      Vector4Float[2].z=ViewProjectFrustumVec[iFrustumIndex,e,2]*(iPosZ+iSizeZ)
      rCull=0
      if Vector4Float[1].x+Vector4Float[1].y+Vector4Float[1].z+ViewProjectFrustumVec[iFrustumIndex,e,3]>0 then rCull=1
      if rCull=0
         if Vector4Float[2].x+Vector4Float[1].y+Vector4Float[1].z+ViewProjectFrustumVec[iFrustumIndex,e,3]>0 then rCull=1
         if Vector4Float[1].x+Vector4Float[2].y+Vector4Float[1].z+ViewProjectFrustumVec[iFrustumIndex,e,3]>0 then rCull=1
         if Vector4Float[2].x+Vector4Float[2].y+Vector4Float[1].z+ViewProjectFrustumVec[iFrustumIndex,e,3]>0 then rCull=1
      endif
      if rCull=0
         if Vector4Float[1].x+Vector4Float[1].y+Vector4Float[2].z+ViewProjectFrustumVec[iFrustumIndex,e,3]>0 then rCull=1
         if Vector4Float[2].x+Vector4Float[1].y+Vector4Float[2].z+ViewProjectFrustumVec[iFrustumIndex,e,3]>0 then rCull=1
         if Vector4Float[1].x+Vector4Float[2].y+Vector4Float[2].z+ViewProjectFrustumVec[iFrustumIndex,e,3]>0 then rCull=1
         if Vector4Float[2].x+Vector4Float[2].y+Vector4Float[2].z+ViewProjectFrustumVec[iFrustumIndex,e,3]>0 then rCull=1
      endif
      if rCull=0 then exitfunction 0
   next e
endfunction 1

`--------------
` Set Vectors
`--------------
function Set_Vector2(iVector as integer,iX as float,iY as float)
   Vector2Float[iVector].x=iX
   Vector2Float[iVector].y=iY
endfunction
function Set_Vector3(iVector as integer,iX as float,iY as float,iZ as float)
   Vector3Float[iVector].x=iX
   Vector3Float[iVector].y=iY
   Vector3Float[iVector].z=iZ
endfunction
function Set_Vector4(iVector as integer,iX as float,iY as float,iZ as float,iw as float)
   Vector4Float[iVector].x=iX
   Vector4Float[iVector].y=iY
   Vector4Float[iVector].z=iZ
   Vector4Float[iVector].w=iw
endfunction

`--------------
` Get Vectors
`--------------
function X_Vector2(iVector as integer)
   r#=Vector2Float[iVector].x
endfunction r#
function Y_Vector2(iVector as integer)
   r#=Vector2Float[iVector].y
endfunction r#
function X_Vector3(iVector as integer)
   r#=Vector3Float[iVector].x
endfunction r#
function Y_Vector3(iVector as integer)
   r#=Vector3Float[iVector].y
endfunction r#
function Z_Vector3(iVector as integer)
   r#=Vector3Float[iVector].z
endfunction r#
function X_Vector4(iVector as integer)
   r#=Vector4Float[iVector].x
endfunction r#
function Y_Vector4(iVector as integer)
   r#=Vector4Float[iVector].y
endfunction r#
function Z_Vector4(iVector as integer)
   r#=Vector4Float[iVector].z
endfunction r#
function w_Vector4(iVector as integer)
   r#=Vector4Float[iVector].w
endfunction r#

`--------------
` Add Vectors
`--------------
function Add_Vector2(iVectorResult as integer,iVectorA as integer,iVectorB as integer)
   Vector2Float[iVectorResult].x=Vector2Float[iVectorA].x+Vector2Float[iVectorB].x
   Vector2Float[iVectorResult].y=Vector2Float[iVectorA].y+Vector2Float[iVectorB].y
endfunction
function Add_Vector3(iVectorResult as integer,iVectorA as integer,iVectorB as integer)
   Vector3Float[iVectorResult].x=Vector3Float[iVectorA].x+Vector3Float[iVectorB].x
   Vector3Float[iVectorResult].y=Vector3Float[iVectorA].y+Vector3Float[iVectorB].y
   Vector3Float[iVectorResult].z=Vector3Float[iVectorA].z+Vector3Float[iVectorB].z
endfunction
function Add_Vector4(iVectorResult as integer,iVectorA as integer,iVectorB as integer)
   Vector4Float[iVectorResult].x=Vector4Float[iVectorA].x+Vector4Float[iVectorB].x
   Vector4Float[iVectorResult].y=Vector4Float[iVectorA].y+Vector4Float[iVectorB].y
   Vector4Float[iVectorResult].z=Vector4Float[iVectorA].z+Vector4Float[iVectorB].z
   Vector4Float[iVectorResult].w=Vector4Float[iVectorA].w+Vector4Float[iVectorB].w
endfunction

`--------------
` Subtract Vectors
`--------------
function Subtract_Vector2(iVectorResult as integer,iVectorA as integer,iVectorB as integer)
   Vector2Float[iVectorResult].x=Vector2Float[iVectorA].x-Vector2Float[iVectorB].x
   Vector2Float[iVectorResult].y=Vector2Float[iVectorA].y-Vector2Float[iVectorB].y
endfunction
function Subtract_Vector3(iVectorResult as integer,iVectorA as integer,iVectorB as integer)
   Vector3Float[iVectorResult].x=Vector3Float[iVectorA].x-Vector3Float[iVectorB].x
   Vector3Float[iVectorResult].y=Vector3Float[iVectorA].y-Vector3Float[iVectorB].y
   Vector3Float[iVectorResult].z=Vector3Float[iVectorA].z-Vector3Float[iVectorB].z
endfunction
function Subtract_Vector4(iVectorResult as integer,iVectorA as integer,iVectorB as integer)
   Vector4Float[iVectorResult].x=Vector4Float[iVectorA].x-Vector4Float[iVectorB].x
   Vector4Float[iVectorResult].y=Vector4Float[iVectorA].y-Vector4Float[iVectorB].y
   Vector4Float[iVectorResult].z=Vector4Float[iVectorA].z-Vector4Float[iVectorB].z
   Vector4Float[iVectorResult].w=Vector4Float[iVectorA].w-Vector4Float[iVectorB].w
endfunction

`--------------
` Multiply Vectors
`--------------
function Multiply_Vector2(iVector as integer,iValue as float)
   Vector2Float[iVector].x=Vector2Float[iVector].x*iValue
   Vector2Float[iVector].y=Vector2Float[iVector].y*iValue
endfunction
function Multiply_Vector3(iVector as integer,iValue as float)
   Vector3Float[iVector].x=Vector3Float[iVector].x*iValue
   Vector3Float[iVector].y=Vector3Float[iVector].y*iValue
   Vector3Float[iVector].z=Vector3Float[iVector].z*iValue
endfunction
function Multiply_Vector4(iVector as integer,iValue as float)
   Vector4Float[iVector].x=Vector4Float[iVector].x*iValue
   Vector4Float[iVector].y=Vector4Float[iVector].y*iValue
   Vector4Float[iVector].z=Vector4Float[iVector].z*iValue
   Vector4Float[iVector].w=Vector4Float[iVector].w*iValue
endfunction

`--------------
` Divide Vectors
`--------------
function Divide_Vector2(iVector as integer,iValue as float)
   Vector2Float[iVector].x=Vector2Float[iVector].x/iValue
   Vector2Float[iVector].y=Vector2Float[iVector].y/iValue
endfunction
function Divide_Vector3(iVector as integer,iValue as float)
   Vector3Float[iVector].x=Vector3Float[iVector].x/iValue
   Vector3Float[iVector].y=Vector3Float[iVector].y/iValue
   Vector3Float[iVector].z=Vector3Float[iVector].z/iValue
endfunction
function Divide_Vector4(iVector as integer,iValue as float)
   Vector4Float[iVector].x=Vector4Float[iVector].x/iValue
   Vector4Float[iVector].y=Vector4Float[iVector].y/iValue
   Vector4Float[iVector].z=Vector4Float[iVector].z/iValue
   Vector4Float[iVector].w=Vector4Float[iVector].w/iValue
endfunction

`--------------
` Length Vectors
`--------------
function Length_Vector2(iVector as integer)
   r#=sqrt((Vector4Float[iVector].x^2)+(Vector4Float[iVector].y^2))
endfunction r#
function Length_Vector3(iVector as integer)
   r#=sqrt((Vector3Float[iVector].x^2)+(Vector3Float[iVector].y^2)+(Vector3Float[iVector].z^2))
endfunction r#
function Length_Vector4(iVector as integer)
   r#=sqrt((Vector4Float[iVector].x^2)+(Vector4Float[iVector].y^2)+(Vector4Float[iVector].z^2)+(Vector4Float[iVector].w^2))
endfunction r#

`--------------
` Dot Product Vectors
`--------------
function Dot_Product_Vector2(iVectorA as integer,iVectorB as integer)
   r#=Vector2Float[iVectorA].x*Vector2Float[iVectorB].x+(Vector2Float[iVectorA].y*Vector2Float[iVectorB].y)
endfunction r#
function Dot_Product_Vector3(iVectorA as integer,iVectorB as integer)
   r#=Vector3Float[iVectorA].x*Vector3Float[iVectorB].x+(Vector3Float[iVectorA].y*Vector3Float[iVectorB].y)+(Vector3Float[iVectorA].z*Vector3Float[iVectorB].z)
endfunction r#
function Dot_Product_Vector4(iVectorA as integer,iVectorB as integer)
   r#=Vector4Float[iVectorA].x*Vector4Float[iVectorB].x+(Vector4Float[iVectorA].y*Vector4Float[iVectorB].y)+(Vector4Float[iVectorA].z*Vector4Float[iVectorB].z)+(Vector4Float[iVectorA].w*Vector4Float[iVectorB].w)
endfunction r#

`--------------
` Normalize Vectors
`--------------
function Normalize_Vector2(iVectorResult as integer,iVector as integer)
   L#=sqrt((Vector4Float[iVector].x^2)+(Vector4Float[iVector].y^2))
   Vector2Float[iVectorResult].x=Vector4Float[iVector].x/L#
   Vector2Float[iVectorResult].y=Vector4Float[iVector].y/L#
endfunction
function Normalize_Vector3(iVectorResult as integer,iVector as integer)
   L#=sqrt((Vector3Float[iVector].x^2)+(Vector3Float[iVector].y^2)+(Vector3Float[iVector].z^2))
   Vector3Float[iVectorResult].x=Vector3Float[iVector].x/L#
   Vector3Float[iVectorResult].y=Vector3Float[iVector].y/L#
   Vector3Float[iVectorResult].z=Vector3Float[iVector].z/L#
endfunction
function Normalize_Vector4(iVectorResult as integer,iVector as integer)
   L#=sqrt((Vector4Float[iVector].x^2)+(Vector4Float[iVector].y^2)+(Vector4Float[iVector].z^2)+(Vector4Float[iVector].w^2))
   Vector4Float[iVectorResult].x=Vector4Float[iVector].x/L#
   Vector4Float[iVectorResult].y=Vector4Float[iVector].y/L#
   Vector4Float[iVectorResult].z=Vector4Float[iVector].z/L#
   Vector4Float[iVectorResult].w=Vector4Float[iVector].w/L#
endfunction

`--------------
` Linear Interpolate Vectors
`--------------
function Linear_Interpolate_Vector2(iVectorResult as integer,iVectorA as integer,iVectorB as integer,iValue as float)
   Vector2Float[iVectorResult].x=Vector2Float[iVectorA].x+iValue*(Vector2Float[iVectorB].x-Vector2Float[iVectorA].x)
   Vector2Float[iVectorResult].y=Vector2Float[iVectorA].y+iValue*(Vector2Float[iVectorB].y-Vector2Float[iVectorA].y)
endfunction
function Linear_Interpolate_Vector3(iVectorResult as integer,iVectorA as integer,iVectorB as integer,iValue as float)
   Vector3Float[iVectorResult].x=Vector3Float[iVectorA].x+iValue*(Vector3Float[iVectorB].x-Vector3Float[iVectorA].x)
   Vector3Float[iVectorResult].y=Vector3Float[iVectorA].y+iValue*(Vector3Float[iVectorB].y-Vector3Float[iVectorA].y)
   Vector3Float[iVectorResult].z=Vector3Float[iVectorA].z+iValue*(Vector3Float[iVectorB].z-Vector3Float[iVectorA].z)
endfunction
function Linear_Interpolate_Vector4(iVectorResult as integer,iVectorA as integer,iVectorB as integer,iValue as float)
   Vector4Float[iVectorResult].x=Vector4Float[iVectorA].x+iValue*(Vector4Float[iVectorB].x-Vector4Float[iVectorA].x)
   Vector4Float[iVectorResult].y=Vector4Float[iVectorA].y+iValue*(Vector4Float[iVectorB].y-Vector4Float[iVectorA].y)
   Vector4Float[iVectorResult].z=Vector4Float[iVectorA].z+iValue*(Vector4Float[iVectorB].z-Vector4Float[iVectorA].z)
   Vector4Float[iVectorResult].w=Vector4Float[iVectorA].w+iValue*(Vector4Float[iVectorB].w-Vector4Float[iVectorA].w)
endfunction

`--------------
` Barycentric Coordinates Vectors
`--------------
function BCC_Vector2(iVectorResult as integer,iVectorA as integer,iVectorB as integer,iVectorC as integer,iFValue as float,iGBValue as float)
   Subtract_Vector2(257,iVectorB,iVectorA)
   Subtract_Vector2(258,iVectorC,iVectorA)
   Multiply_Vector2(257,iFValue)
   Multiply_Vector2(258,iGBValue)
   Add_Vector2(iVectorResult,257,258)
   Add_Vector2(iVectorResult,iVectorA,iVectorResult)
endfunction
function BCC_Vector3(iVectorResult as integer,iVectorA as integer,iVectorB as integer,iVectorC as integer,iFValue as float,iGBValue as float)
   Subtract_Vector3(257,iVectorB,iVectorA)
   Subtract_Vector3(258,iVectorC,iVectorA)
   Multiply_Vector3(257,iFValue)
   Multiply_Vector3(258,iGBValue)
   Add_Vector3(iVectorResult,257,258)
   Add_Vector3(iVectorResult,iVectorA,iVectorResult)
endfunction
function BCC_Vector4(iVectorResult as integer,iVectorA as integer,iVectorB as integer,iVectorC as integer,iFValue as float,iGBValue as float)
   Subtract_Vector4(257,iVectorB,iVectorA)
   Subtract_Vector4(258,iVectorC,iVectorA)
   Multiply_Vector4(257,iFValue)
   Multiply_Vector4(258,iGBValue)
   Add_Vector4(iVectorResult,257,258)
   Add_Vector4(iVectorResult,iVectorA,iVectorResult)
endfunction

`--------------
` Catmullrom Vectors
`--------------
function CatmullRom_Vector2(iVectorResult as integer,iVectorA as integer,iVectorB as integer,iVectorC as integer,iVectorD as integer,iValue as float)
   XA#=Vector2Float[iVectorA].x : YA#=Vector2Float[iVectorA].y
   XB#=Vector2Float[iVectorB].x : YB#=Vector2Float[iVectorB].y
   XC#=Vector2Float[iVectorC].x : YC#=Vector2Float[iVectorC].y
   XD#=Vector2Float[iVectorD].x : YD#=Vector2Float[iVectorD].y
   Set_Vector2(257,XA#,YA#)
   Set_Vector2(258,XB#,YB#)
   Set_Vector2(259,XC#,YC#)
   Set_Vector2(260,XD#,YD#)
   Multiply_Vector2(257,-1.0*iValue^3+2.0*iValue^2-iValue)
   Multiply_Vector2(258,3.0*iValue^3-5.0*iValue^2+2.0)
   Multiply_Vector2(259,-3.0*iValue^3+4.0*iValue^2+iValue)
   Multiply_Vector2(260,iValue^3-iValue^2)
   Add_Vector2(iVectorResult,257,258)
   Add_Vector2(iVectorResult,iVectorResult,259)
   Add_Vector2(iVectorResult,iVectorResult,260)
   Divide_Vector2(iVectorResult,2.0)
endfunction
function CatmullRom_Vector3(iVectorResult as integer,iVectorA as integer,iVectorB as integer,iVectorC as integer,iVectorD as integer,iValue as float)
   XA#=Vector3Float[iVectorA].x : YA#=Vector3Float[iVectorA].y : ZA#=Vector3Float[iVectorA].z
   XB#=Vector3Float[iVectorB].x : YB#=Vector3Float[iVectorB].y : ZB#=Vector3Float[iVectorB].z
   XC#=Vector3Float[iVectorC].x : YC#=Vector3Float[iVectorC].y : zC#=Vector3Float[iVectorC].z
   XD#=Vector3Float[iVectorD].x : YD#=Vector3Float[iVectorD].y : ZD#=Vector3Float[iVectorD].z
   Set_Vector3(257,XA#,YA#,ZA#)
   Set_Vector3(258,XB#,YB#,ZB#)
   Set_Vector3(259,XC#,YC#,ZC#)
   Set_Vector3(260,XD#,YD#,ZD#)
   Multiply_Vector3(257,-1.0*iValue^3+2.0*iValue^2-iValue)
   Multiply_Vector3(258,3.0*iValue^3-5.0*iValue^2+2.0)
   Multiply_Vector3(259,-3.0*iValue^3+4.0*iValue^2+iValue)
   Multiply_Vector3(260,iValue^3-iValue^2)
   Add_Vector3(iVectorResult,257,258)
   Add_Vector3(iVectorResult,iVectorResult,259)
   Add_Vector3(iVectorResult,iVectorResult,260)
   Divide_Vector3(iVectorResult,2.0)
endfunction
function CatmullRom_Vector4(iVectorResult as integer,iVectorA as integer,iVectorB as integer,iVectorC as integer,iVectorD as integer,iValue as float)
   XA#=Vector4Float[iVectorA].x : YA#=Vector4Float[iVectorA].y : ZA#=Vector4Float[iVectorA].z : wA#=Vector4Float[iVectorA].w
   XB#=Vector4Float[iVectorB].x : YB#=Vector4Float[iVectorB].y : ZB#=Vector4Float[iVectorB].z : wB#=Vector4Float[iVectorB].w
   XC#=Vector4Float[iVectorC].x : YC#=Vector4Float[iVectorC].y : zC#=Vector4Float[iVectorC].z : wC#=Vector4Float[iVectorC].w
   XD#=Vector4Float[iVectorD].x : YD#=Vector4Float[iVectorD].y : ZD#=Vector4Float[iVectorD].z : wD#=Vector4Float[iVectorD].w
   Set_Vector4(257,XA#,YA#,ZA#,wA#)
   Set_Vector4(258,XB#,YB#,ZB#,wB#)
   Set_Vector4(259,XC#,YC#,ZC#,wC#)
   Set_Vector4(260,XD#,YD#,ZD#,wD#)
   Multiply_Vector4(257,-1.0*iValue^3+2.0*iValue^2-iValue)
   Multiply_Vector4(258,3.0*iValue^3-5.0*iValue^2+2.0)
   Multiply_Vector4(259,-3.0*iValue^3+4.0*iValue^2+iValue)
   Multiply_Vector4(260,iValue^3-iValue^2)
   Add_Vector4(iVectorResult,257,258)
   Add_Vector4(iVectorResult,iVectorResult,259)
   Add_Vector4(iVectorResult,iVectorResult,260)
   Divide_Vector4(iVectorResult,2.0)
endfunction

`--------------
` Hermite Vectors
`--------------
function Hermite_Vector2(iVectorResult as integer,iVectorA as integer,iVectorB as integer,iVectorC as integer,iVectorD as integer,iValue as float)
   XA#=Vector2Float[iVectorA].x : YA#=Vector2Float[iVectorA].y
   XB#=Vector2Float[iVectorB].x : YB#=Vector2Float[iVectorB].y
   XC#=Vector2Float[iVectorC].x : YC#=Vector2Float[iVectorC].y
   XD#=Vector2Float[iVectorD].x : YD#=Vector2Float[iVectorD].y
   Set_Vector2(257,XA#,YA#)
   Set_Vector2(258,XB#,YB#)
   Set_Vector2(259,XC#,YC#)
   Set_Vector2(260,XD#,YD#)
   Multiply_Vector2(257,2.0*iValue^3-3.0*iValue^2+1.0)
   Multiply_Vector2(258,-2.0*iValue^3+3.0*iValue^2)
   Multiply_Vector2(259,iValue^3-2.0*iValue^2+iValue)
   Multiply_Vector2(260,iValue^3-iValue^2)
   Add_Vector2(iVectorResult,257,258)
   Add_Vector2(iVectorResult,iVectorResult,259)
   Add_Vector2(iVectorResult,iVectorResult,260)
endfunction
function Hermite_Vector3(iVectorResult as integer,iVectorA as integer,iVectorB as integer,iVectorC as integer,iVectorD as integer,iValue as float)
   XA#=Vector3Float[iVectorA].x : YA#=Vector3Float[iVectorA].y : ZA#=Vector3Float[iVectorA].z
   XB#=Vector3Float[iVectorB].x : YB#=Vector3Float[iVectorB].y : ZB#=Vector3Float[iVectorB].z
   XC#=Vector3Float[iVectorC].x : YC#=Vector3Float[iVectorC].y : zC#=Vector3Float[iVectorC].z
   XD#=Vector3Float[iVectorD].x : YD#=Vector3Float[iVectorD].y : ZD#=Vector3Float[iVectorD].z
   Set_Vector3(257,XA#,YA#,ZA#)
   Set_Vector3(258,XB#,YB#,ZB#)
   Set_Vector3(259,XC#,YC#,ZC#)
   Set_Vector3(260,XD#,YD#,ZD#)
   Multiply_Vector3(257,2.0*iValue^3-3.0*iValue^2+1.0)
   Multiply_Vector3(258,-2.0*iValue^3+3.0*iValue^2)
   Multiply_Vector3(259,iValue^3-2.0*iValue^2+iValue)
   Multiply_Vector3(260,iValue^3-iValue^2)
   Add_Vector3(iVectorResult,257,258)
   Add_Vector3(iVectorResult,iVectorResult,259)
   Add_Vector3(iVectorResult,iVectorResult,260)
endfunction
function Hermite_Vector4(iVectorResult as integer,iVectorA as integer,iVectorB as integer,iVectorC as integer,iVectorD as integer,iValue as float)
   XA#=Vector4Float[iVectorA].x : YA#=Vector4Float[iVectorA].y : ZA#=Vector4Float[iVectorA].z : wA#=Vector4Float[iVectorA].w
   XB#=Vector4Float[iVectorB].x : YB#=Vector4Float[iVectorB].y : ZB#=Vector4Float[iVectorB].z : wB#=Vector4Float[iVectorB].w
   XC#=Vector4Float[iVectorC].x : YC#=Vector4Float[iVectorC].y : zC#=Vector4Float[iVectorC].z : wC#=Vector4Float[iVectorC].w
   XD#=Vector4Float[iVectorD].x : YD#=Vector4Float[iVectorD].y : ZD#=Vector4Float[iVectorD].z : wD#=Vector4Float[iVectorD].w
   Set_Vector4(257,XA#,YA#,ZA#,wA#)
   Set_Vector4(258,XB#,YB#,ZB#,wB#)
   Set_Vector4(259,XC#,YC#,ZC#,wC#)
   Set_Vector4(260,XD#,YD#,ZD#,wD#)
   Multiply_Vector4(257,2.0*iValue^3-3.0*iValue^2+1.0)
   Multiply_Vector4(258,-2.0*iValue^3+3.0*iValue^2)
   Multiply_Vector4(259,iValue^3-2.0*iValue^2+iValue)
   Multiply_Vector4(260,iValue^3-iValue^2)
   Add_Vector4(iVectorResult,257,258)
   Add_Vector4(iVectorResult,iVectorResult,259)
   Add_Vector4(iVectorResult,iVectorResult,260)
endfunction

`--------------
` Cross Product Vector3
`--------------
function Cross_Product_Vector3(iVectorResult as integer,iVectorA as integer,iVectorB as integer)
   Vector3Float[iVectorResult].x=(Vector3Float[iVectorA].y*Vector3Float[iVectorB].z)-(Vector3Float[iVectorA].z*Vector3Float[iVectorB].y)
   Vector3Float[iVectorResult].y=(Vector3Float[iVectorA].z*Vector3Float[iVectorB].x)-(Vector3Float[iVectorA].x*Vector3Float[iVectorB].z)
   Vector3Float[iVectorResult].z=(Vector3Float[iVectorA].x*Vector3Float[iVectorB].y)-(Vector3Float[iVectorA].y*Vector3Float[iVectorB].x)
endfunction

`--------------
` Transform Coords Vector3
`--------------
function Transform_Coords_Vector3(iVectorResult as integer,iVectorSource as integer,iMatrix4Source as integer)
   X#=Vector3Float[iVectorSource].x : Y#=Vector3Float[iVectorSource].y : Z#=Vector3Float[iVectorSource].z
   nX#=X#*Matrix4Float[iMatrix4Source,1].x+Y#*Matrix4Float[iMatrix4Source,2].x+Z#*Matrix4Float[iMatrix4Source,3].x+Matrix4Float[iMatrix4Source,4].x
   nY#=X#*Matrix4Float[iMatrix4Source,1].y+Y#*Matrix4Float[iMatrix4Source,2].y+Z#*Matrix4Float[iMatrix4Source,3].y+Matrix4Float[iMatrix4Source,4].y
   nZ#=X#*Matrix4Float[iMatrix4Source,1].z+Y#*Matrix4Float[iMatrix4Source,2].z+Z#*Matrix4Float[iMatrix4Source,3].z+Matrix4Float[iMatrix4Source,4].z
   nW#=X#*Matrix4Float[iMatrix4Source,1].w+Y#*Matrix4Float[iMatrix4Source,2].w+Z#*Matrix4Float[iMatrix4Source,3].w+Matrix4Float[iMatrix4Source,4].w
   Vector3Float[iVectorResult].x=nX#*(1.0/nW#)
   Vector3Float[iVectorResult].y=nY#*(1.0/nW#)
   Vector3Float[iVectorResult].z=nZ#*(1.0/nW#)
endfunction

`--------------
` Transform Normals Vector3
`--------------
function Transform_Normals_Vector3(iVectorResult as integer,iVectorSource as integer,iMatrix4Source as integer)
   X#=Vector3Float[iVectorSource].x : Y#=Vector3Float[iVectorSource].y : Z#=Vector3Float[iVectorSource].z
   nX#=X#*Matrix4Float[iMatrix4Source,1].x+Y#*Matrix4Float[iMatrix4Source,2].x+Z#*Matrix4Float[iMatrix4Source,3].x
   nY#=X#*Matrix4Float[iMatrix4Source,1].y+Y#*Matrix4Float[iMatrix4Source,2].y+Z#*Matrix4Float[iMatrix4Source,3].y
   nZ#=X#*Matrix4Float[iMatrix4Source,1].z+Y#*Matrix4Float[iMatrix4Source,2].z+Z#*Matrix4Float[iMatrix4Source,3].z
   Vector3Float[iVectorResult].x=nX#
   Vector3Float[iVectorResult].y=nY#
   Vector3Float[iVectorResult].z=nZ#
endfunction

`--------------
` Project Vector3
`--------------
function Project_Vector3(iVectorResult as integer,iVectorSource as integer,iMatrix4Projection as integer,iMatrix4View as integer,iMatrix4World as integer)
   Transform_Coords_Vector3(260,iVectorSource,iMatrix4World)
   Transform_Coords_Vector3(260,260,iMatrix4View)
   Transform_Coords_Vector3(iVectorResult,260,iMatrix4Projection)
   Vector3Float[iVectorResult].x=(Vector3Float[iVectorResult].x+1.0)*(GetVirtualWidth()/2.0)
   Vector3Float[iVectorResult].y=(1.0-Vector3Float[iVectorResult].y)*(GetVirtualHeight()/2.0)
   Vector3Float[iVectorResult].z=Vector3Float[iVectorResult].z
endfunction

`--------------
` Transform Vector4
`--------------
function Transform_Vector4(iVectorResult as integer,iVectorSource as integer,iMatrix4Source as integer)
   X#=Vector4Float[iVectorSource].x : Y#=Vector4Float[iVectorSource].y : Z#=Vector4Float[iVectorSource].z : W#=Vector4Float[iVectorSource].w
   nX#=X#*Matrix4Float[iMatrix4Source,1].x+Y#*Matrix4Float[iMatrix4Source,2].x+Z#*Matrix4Float[iMatrix4Source,3].x+W#*Matrix4Float[iMatrix4Source,4].x
   nY#=X#*Matrix4Float[iMatrix4Source,1].y+Y#*Matrix4Float[iMatrix4Source,2].y+Z#*Matrix4Float[iMatrix4Source,3].y+W#*Matrix4Float[iMatrix4Source,4].y
   nZ#=X#*Matrix4Float[iMatrix4Source,1].z+Y#*Matrix4Float[iMatrix4Source,2].z+Z#*Matrix4Float[iMatrix4Source,3].z+W#*Matrix4Float[iMatrix4Source,4].z
   nW#=X#*Matrix4Float[iMatrix4Source,1].w+Y#*Matrix4Float[iMatrix4Source,2].w+Z#*Matrix4Float[iMatrix4Source,3].w+W#*Matrix4Float[iMatrix4Source,4].w
   Vector4Float[iVectorResult].x=nX#
   Vector4Float[iVectorResult].y=nY#
   Vector4Float[iVectorResult].z=nZ#
   Vector4Float[iVectorResult].w=nW#
endfunction

`--------------
` Elements Matrix4
`--------------
function SetElement_Matrix4(iMatrix as integer,iElement as integer,iValue as float)
   Element=((iElement-1)/4) : Idx=iElement-(Element*4)
   if Idx=1 then Matrix4Float[iMatrix,Element+1].x=iValue
   if Idx=2 then Matrix4Float[iMatrix,Element+1].y=iValue
   if Idx=3 then Matrix4Float[iMatrix,Element+1].z=iValue
   if Idx=4 then Matrix4Float[iMatrix,Element+1].w=iValue
endfunction
function GetElement_Matrix4(iMatrix as integer,iElement as integer)
   Element=((iElement-1)/4) : Idx=iElement-(Element*4)
   if Idx=1 then r#=Matrix4Float[1,Element+1].x
   if Idx=2 then r#=Matrix4Float[1,Element+1].y
   if Idx=3 then r#=Matrix4Float[1,Element+1].z
   if Idx=4 then r#=Matrix4Float[1,Element+1].w
endfunction r#

`--------------
` Fill Matrix4
`--------------
function Fill_Matrix4(iMatrixResult as integer,iValue as float)
   for i=1 to 4
      Matrix4Float[iMatrixResult,i].x=iValue
      Matrix4Float[iMatrixResult,i].y=iValue
      Matrix4Float[iMatrixResult,i].z=iValue
      Matrix4Float[iMatrixResult,i].w=iValue
   next i
endfunction

`--------------
` Copy Matrix4
`--------------
function Copy_Matrix4(iMatrixResult as integer,iMatrixSource as integer)
   for i=1 to 4
      Matrix4Float[iMatrixResult,i].x=Matrix4Float[iMatrixSource,i].x
      Matrix4Float[iMatrixResult,i].y=Matrix4Float[iMatrixSource,i].y
      Matrix4Float[iMatrixResult,i].z=Matrix4Float[iMatrixSource,i].z
      Matrix4Float[iMatrixResult,i].w=Matrix4Float[iMatrixSource,i].w
   next i
endfunction

`--------------
` Add Matrix4
`--------------
function Add_Matrix4(iMatrixResult as integer,iMatrixA as integer,iMatrixB as integer)
   for i=1 to 4
      Matrix4Float[iMatrixResult,i].x=Matrix4Float[iMatrixA,i].x+Matrix4Float[iMatrixB,i].x
      Matrix4Float[iMatrixResult,i].y=Matrix4Float[iMatrixA,i].y+Matrix4Float[iMatrixB,i].y
      Matrix4Float[iMatrixResult,i].z=Matrix4Float[iMatrixA,i].z+Matrix4Float[iMatrixB,i].z
      Matrix4Float[iMatrixResult,i].w=Matrix4Float[iMatrixA,i].w+Matrix4Float[iMatrixB,i].w
   next i
endfunction

`--------------
` Subtract Matrix4
`--------------
function Subtract_Matrix4(iMatrixResult as integer,iMatrixA as integer,iMatrixB as integer)
   for i=1 to 4
      Matrix4Float[iMatrixResult,i].x=Matrix4Float[iMatrixA,i].x-Matrix4Float[iMatrixB,i].x
      Matrix4Float[iMatrixResult,i].y=Matrix4Float[iMatrixA,i].y-Matrix4Float[iMatrixB,i].y
      Matrix4Float[iMatrixResult,i].z=Matrix4Float[iMatrixA,i].z-Matrix4Float[iMatrixB,i].z
      Matrix4Float[iMatrixResult,i].w=Matrix4Float[iMatrixA,i].w-Matrix4Float[iMatrixB,i].w
   next i
endfunction

`--------------
` Multiply Matrix4
`--------------
function Multiply_Matrix4(iMatrixResult as integer,iMatrixA as integer,iMatrixB as integer)
   for i=1 to 4
      Matrix4Float[257,i].x=Matrix4Float[iMatrixA,i].x*Matrix4Float[iMatrixB,1].x+Matrix4Float[iMatrixA,i].y*Matrix4Float[iMatrixB,2].x+Matrix4Float[iMatrixA,i].z*Matrix4Float[iMatrixB,3].x+Matrix4Float[iMatrixA,i].w*Matrix4Float[iMatrixB,4].x
      Matrix4Float[257,i].y=Matrix4Float[iMatrixA,i].x*Matrix4Float[iMatrixB,1].y+Matrix4Float[iMatrixA,i].y*Matrix4Float[iMatrixB,2].y+Matrix4Float[iMatrixA,i].z*Matrix4Float[iMatrixB,3].y+Matrix4Float[iMatrixA,i].w*Matrix4Float[iMatrixB,4].y
      Matrix4Float[257,i].z=Matrix4Float[iMatrixA,i].x*Matrix4Float[iMatrixB,1].z+Matrix4Float[iMatrixA,i].y*Matrix4Float[iMatrixB,2].z+Matrix4Float[iMatrixA,i].z*Matrix4Float[iMatrixB,3].z+Matrix4Float[iMatrixA,i].w*Matrix4Float[iMatrixB,4].z
      Matrix4Float[257,i].w=Matrix4Float[iMatrixA,i].x*Matrix4Float[iMatrixB,1].w+Matrix4Float[iMatrixA,i].y*Matrix4Float[iMatrixB,2].w+Matrix4Float[iMatrixA,i].z*Matrix4Float[iMatrixB,3].w+Matrix4Float[iMatrixA,i].w*Matrix4Float[iMatrixB,4].w
   next i
   Copy_Matrix4(iMatrixResult,257)
endfunction

`--------------
` Set Identity Matrix4
`--------------
function SetIdentity_Matrix4(iMatrixResult as integer)
   Fill_Matrix4(iMatrixResult,0.0)
   Matrix4Float[iMatrixResult,1].x=1.0
   Matrix4Float[iMatrixResult,2].y=1.0
   Matrix4Float[iMatrixResult,3].z=1.0
   Matrix4Float[iMatrixResult,4].w=1.0
endfunction

`--------------
` Build Look At Matrix4
`--------------
function Build_LookAtLH_Matrix4(iMatrixResult as integer,iVectorEye as integer,iVectorAt as integer,iVectorUp as integer)
   Subtract_Vector3(257,iVectorAt,iVectorEye)
   Normalize_Vector3(257,257)
   Cross_Product_Vector3(258,iVectorUp,257)
   Normalize_Vector3(258,258)
   Cross_Product_Vector3(259,257,258)
   Matrix4Float[iMatrixResult,1].x=Vector3Float[258].x
   Matrix4Float[iMatrixResult,1].y=Vector3Float[259].x
   Matrix4Float[iMatrixResult,1].z=Vector3Float[257].x
   Matrix4Float[iMatrixResult,1].w=0.0
   Matrix4Float[iMatrixResult,2].x=Vector3Float[258].y
   Matrix4Float[iMatrixResult,2].y=Vector3Float[259].y
   Matrix4Float[iMatrixResult,2].z=Vector3Float[257].y
   Matrix4Float[iMatrixResult,2].w=0.0
   Matrix4Float[iMatrixResult,3].x=Vector3Float[258].z
   Matrix4Float[iMatrixResult,3].y=Vector3Float[259].z
   Matrix4Float[iMatrixResult,3].z=Vector3Float[257].z
   Matrix4Float[iMatrixResult,3].w=0.0
   Matrix4Float[iMatrixResult,4].x=-1.0*Dot_Product_Vector3(258,iVectorEye)
   Matrix4Float[iMatrixResult,4].y=-1.0*Dot_Product_Vector3(259,iVectorEye)
   Matrix4Float[iMatrixResult,4].z=-1.0*Dot_Product_Vector3(257,iVectorEye)
   Matrix4Float[iMatrixResult,4].w=1.0
endfunction
function Build_LookAtRH_Matrix4(iMatrixResult as integer,iVectorEye as integer,iVectorAt as integer,iVectorUp as integer)
   Subtract_Vector3(257,iVectorEye,iVectorAt)
   Normalize_Vector3(257,257)
   Cross_Product_Vector3(258,iVectorUp,257)
   Normalize_Vector3(258,258)
   Cross_Product_Vector3(259,257,258)
   Matrix4Float[iMatrixResult,1].x=Vector3Float[258].x
   Matrix4Float[iMatrixResult,1].y=Vector3Float[259].x
   Matrix4Float[iMatrixResult,1].z=Vector3Float[257].x
   Matrix4Float[iMatrixResult,1].w=0.0
   Matrix4Float[iMatrixResult,2].x=Vector3Float[258].y
   Matrix4Float[iMatrixResult,2].y=Vector3Float[259].y
   Matrix4Float[iMatrixResult,2].z=Vector3Float[257].y
   Matrix4Float[iMatrixResult,2].w=0.0
   Matrix4Float[iMatrixResult,3].x=Vector3Float[258].z
   Matrix4Float[iMatrixResult,3].y=Vector3Float[259].z
   Matrix4Float[iMatrixResult,3].z=Vector3Float[257].z
   Matrix4Float[iMatrixResult,3].w=0.0
   Matrix4Float[iMatrixResult,4].x=-1.0*Dot_Product_Vector3(258,iVectorEye)
   Matrix4Float[iMatrixResult,4].y=-1.0*Dot_Product_Vector3(259,iVectorEye)
   Matrix4Float[iMatrixResult,4].z=-1.0*Dot_Product_Vector3(257,iVectorEye)
   Matrix4Float[iMatrixResult,4].w=1.0
endfunction

`--------------
` Build Ortho Matrix4
`--------------
function Build_OrthoLH_Matrix4(iMatrixResult as integer,iWidth as float,iHeight as float,iNear as float,iFar as float)
   Fill_Matrix4(iMatrixResult,0.0)
   Matrix4Float[iMatrixResult,1].x=2.0/iWidth
   Matrix4Float[iMatrixResult,2].y=2.0/iHeight
   Matrix4Float[iMatrixResult,3].z=1.0/(iFar-iNear)
   Matrix4Float[iMatrixResult,4].z=iNear/(iNear-iFar)
   Matrix4Float[iMatrixResult,4].w=1.0
endfunction
function Build_OrthoRH_Matrix4(iMatrixResult as integer,iWidth as float,iHeight as float,iNear as float,iFar as float)
   Fill_Matrix4(iMatrixResult,0.0)
   Matrix4Float[iMatrixResult,1].x=2.0/iWidth
   Matrix4Float[iMatrixResult,2].y=2.0/iHeight
   Matrix4Float[iMatrixResult,3].z=1.0/(iNear-iFar)
   Matrix4Float[iMatrixResult,4].z=iNear/(iNear-iFar)
   Matrix4Float[iMatrixResult,4].w=1.0
endfunction

`--------------
` Build FOV Matrix4
`--------------
function Build_FOVLH_Matrix4(iMatrixResult as integer,iFOV as float,iAspect as float,iNear as float,iFar as float)
   FOV#=1.0/tan(iFOV/2.0)
   Fill_Matrix4(iMatrixResult,0.0)
   Matrix4Float[iMatrixResult,1].x=FOV#/iAspect
   Matrix4Float[iMatrixResult,2].y=FOV#
   Matrix4Float[iMatrixResult,3].z=iFar/(iFar-iNear)
   Matrix4Float[iMatrixResult,3].w=1.0
   Matrix4Float[iMatrixResult,4].z=-1.0*iNear*(iFar/(iFar-iNear))
endfunction
function Build_FOVRH_Matrix4(iMatrixResult as integer,iFOV as float,iAspect as float,iNear as float,iFar as float)
   FOV#=1.0/tan(iFOV/2.0)
   Fill_Matrix4(iMatrixResult,0.0)
   Matrix4Float[iMatrixResult,1].x=FOV#/iAspect
   Matrix4Float[iMatrixResult,2].y=FOV#
   Matrix4Float[iMatrixResult,3].z=1.0*iFar/(iNear-iFar)
   Matrix4Float[iMatrixResult,3].w=-1.0
   Matrix4Float[iMatrixResult,4].z=1.0*iNear*iFar/(iNear-iFar)
endfunction

`--------------
` Build Perspective Matrix4
`--------------
function Build_PerspectiveLH_Matrix4(iMatrixResult as integer,iWidth as float,iHeight as float,iNear as float,iFar as float)
   Fill_Matrix4(iMatrixResult,0.0)
   Matrix4Float[iMatrixResult,1].x=2.0*iNear/iWidth
   Matrix4Float[iMatrixResult,2].y=2.0*iNear/iHeight
   Matrix4Float[iMatrixResult,3].z=1.0*iFar/(iFar-iNear)
   Matrix4Float[iMatrixResult,3].w=1.0
   Matrix4Float[iMatrixResult,4].z=1.0*iNear*iFar/(iNear-iFar)
endfunction
function Build_PerspectiveRH_Matrix4(iMatrixResult as integer,iWidth as float,iHeight as float,iNear as float,iFar as float)
   Fill_Matrix4(iMatrixResult,0.0)
   Matrix4Float[iMatrixResult,1].x=2.0*iNear/iWidth
   Matrix4Float[iMatrixResult,2].y=2.0*iNear/iHeight
   Matrix4Float[iMatrixResult,3].z=1.0*iFar/(iNear-iFar)
   Matrix4Float[iMatrixResult,3].w=-1.0
   Matrix4Float[iMatrixResult,4].z=1.0*iNear*iFar/(iNear-iFar)
endfunction

`--------------
` Build Rotation Axis Matrix4
`--------------
function Build_RotationAxis_Matrix4(iMatrixResult as integer,iVectorAxis as integer,iAngle as float)
   iAngle#=iAngle/0.01745329251
   Cos#=Cos(iAngle)
   Sin#=Sin(iAngle)
   t#=1.0-Cos#
   Length#=Length_Vector3(iVectorAxis)
   U#=Vector3Float[iVectorAxis].x/Length#
   V#=Vector3Float[iVectorAxis].y/Length#
   W#=Vector3Float[iVectorAxis].z/Length#
   Matrix4Float[iMatrixResult,1].x=Cos#+U#*U#*t#
   Matrix4Float[iMatrixResult,1].y=W#*Sin#+V#*U#*t#
   Matrix4Float[iMatrixResult,1].z=W#*U#*t#-V#*Sin#
   Matrix4Float[iMatrixResult,1].w=0.0
   Matrix4Float[iMatrixResult,2].x=U#*V#*t#-W#*Sin#
   Matrix4Float[iMatrixResult,2].y=Cos#+V#*V#*t#
   Matrix4Float[iMatrixResult,2].z=U#*Sin#+W#*V#*t#
   Matrix4Float[iMatrixResult,2].w=0.0
   Matrix4Float[iMatrixResult,3].x=V#*Sin#+U#*W#*t#
   Matrix4Float[iMatrixResult,3].y=V#*W#*t#-U#*Sin#
   Matrix4Float[iMatrixResult,3].z=Cos#+W#*W#*t#
   Matrix4Float[iMatrixResult,3].w=0.0
   Matrix4Float[iMatrixResult,4].x=0.0
   Matrix4Float[iMatrixResult,4].y=0.0
   Matrix4Float[iMatrixResult,4].z=0.0
   Matrix4Float[iMatrixResult,4].w=1.0
endfunction

`--------------
` Rotate Matrix4
`--------------
function RotateX_Matrix4(iMatrixResult as integer,iAngle as float)
   iAngle=iAngle/0.01745329251
   SetIdentity_Matrix4(iMatrixResult)
   Matrix4Float[iMatrixResult,2].y=Cos(iAngle)
   Matrix4Float[iMatrixResult,2].z=Sin(iAngle)
   Matrix4Float[iMatrixResult,3].y=-1.0*Sin(iAngle)
   Matrix4Float[iMatrixResult,3].z=Cos(iAngle)
endfunction
function RotateY_Matrix4(iMatrixResult as integer,iAngle as float)
   iAngle=iAngle/0.01745329251
   SetIdentity_Matrix4(iMatrixResult)
   Matrix4Float[iMatrixResult,1].x=Cos(iAngle)
   Matrix4Float[iMatrixResult,1].z=-1.0*Sin(iAngle)
   Matrix4Float[iMatrixResult,3].x=Sin(iAngle)
   Matrix4Float[iMatrixResult,3].z=Cos(iAngle)
endfunction
function RotateZ_Matrix4(iMatrixResult as integer,iAngle as float)
   iAngle=iAngle/0.01745329251
   SetIdentity_Matrix4(iMatrixResult)
   Matrix4Float[iMatrixResult,1].x=Cos(iAngle)
   Matrix4Float[iMatrixResult,1].y=Sin(iAngle)
   Matrix4Float[iMatrixResult,2].x=-1.0*Sin(iAngle)
   Matrix4Float[iMatrixResult,2].y=Cos(iAngle)
endfunction

`--------------
` Rotate Yaw Pitch Roll Matrix4
`--------------
function RotateYPR_Matrix4(iMatrixResult as integer,iYaw as float,iPitch as float,iRoll as float)
   iYaw=iYaw/0.01745329251
   iPitch=iPitch/0.01745329251
   iRoll=iRoll/0.01745329251
   CX#=Cos(iPitch) : SX#=Sin(iPitch)
   CY#=Cos(iYaw) : SY#=Sin(iYaw)
   CZ#=Cos(iRoll) : SZ#=Sin(iRoll)
   SetIdentity_Matrix4(257)
   SetIdentity_Matrix4(258)
   SetIdentity_Matrix4(259)
   SetIdentity_Matrix4(260)
   Matrix4Float[257,2].y=CX#
   Matrix4Float[257,2].z=SX#
   Matrix4Float[257,3].y=-1.0*SX#
   Matrix4Float[257,3].z=CX#
   Matrix4Float[258,1].x=CY#
   Matrix4Float[258,1].z=-1.0*SY#
   Matrix4Float[258,3].x=SY#
   Matrix4Float[258,3].z=CY#
   Matrix4Float[259,1].x=CZ#
   Matrix4Float[259,1].y=SZ#
   Matrix4Float[259,2].x=-1.0*SZ#
   Matrix4Float[259,2].y=CZ#
   Multiply_Matrix4(260,259,257)
   Multiply_Matrix4(iMatrixResult,260,258)
endfunction

`--------------
` Scale Matrix4
`--------------
function Scale_Matrix4(iMatrixResult as integer,iX as float,iY as float,iZ as float)
   SetIdentity_Matrix4(iMatrixResult)
   Matrix4Float[iMatrixResult,1].x=iX
   Matrix4Float[iMatrixResult,2].y=iY
   Matrix4Float[iMatrixResult,3].z=iZ
endfunction

`--------------
` Translate Matrix4
`--------------
function Translate_Matrix4(iMatrixResult as integer,iX as float,iY as float,iZ as float)
   SetIdentity_Matrix4(iMatrixResult)
   Matrix4Float[iMatrixResult,4].x=iX
   Matrix4Float[iMatrixResult,4].y=iY
   Matrix4Float[iMatrixResult,4].z=iZ
endfunction

`--------------
` Transpose Matrix4
`--------------
function Transpose_Matrix4(iMatrixResult as integer,iMatrixSource as integer)
   Matrix4Float[257,1].x=Matrix4Float[iMatrixSource,1].x
   Matrix4Float[257,1].y=Matrix4Float[iMatrixSource,2].x
   Matrix4Float[257,1].z=Matrix4Float[iMatrixSource,3].x
   Matrix4Float[257,1].w=Matrix4Float[iMatrixSource,4].x
   Matrix4Float[257,2].x=Matrix4Float[iMatrixSource,1].y
   Matrix4Float[257,2].y=Matrix4Float[iMatrixSource,2].y
   Matrix4Float[257,2].z=Matrix4Float[iMatrixSource,3].y
   Matrix4Float[257,2].w=Matrix4Float[iMatrixSource,4].y
   Matrix4Float[257,3].x=Matrix4Float[iMatrixSource,1].z
   Matrix4Float[257,3].y=Matrix4Float[iMatrixSource,2].z
   Matrix4Float[257,3].z=Matrix4Float[iMatrixSource,3].z
   Matrix4Float[257,3].w=Matrix4Float[iMatrixSource,4].z
   Matrix4Float[257,4].x=Matrix4Float[iMatrixSource,1].w
   Matrix4Float[257,4].y=Matrix4Float[iMatrixSource,2].w
   Matrix4Float[257,4].z=Matrix4Float[iMatrixSource,3].w
   Matrix4Float[257,4].w=Matrix4Float[iMatrixSource,4].w
   Copy_Matrix4(iMatrixResult,257)
endfunction

`--------------
` Inverse Matrix4
`--------------
function Inverse_Matrix4(iMatrixResult as integer,iMatrixSource as integer)
   Matrix4X1Float[1]=Matrix4Float[iMatrixSource,1].x : Matrix4X1Float[2]=Matrix4Float[iMatrixSource,1].y : Matrix4X1Float[3]=Matrix4Float[iMatrixSource,1].z : Matrix4X1Float[4]=Matrix4Float[iMatrixSource,1].w
   Matrix4X1Float[5]=Matrix4Float[iMatrixSource,2].x : Matrix4X1Float[6]=Matrix4Float[iMatrixSource,2].y : Matrix4X1Float[7]=Matrix4Float[iMatrixSource,2].z : Matrix4X1Float[8]=Matrix4Float[iMatrixSource,2].w
   Matrix4X1Float[9]=Matrix4Float[iMatrixSource,3].x : Matrix4X1Float[10]=Matrix4Float[iMatrixSource,3].y : Matrix4X1Float[11]=Matrix4Float[iMatrixSource,3].z : Matrix4X1Float[12]=Matrix4Float[iMatrixSource,3].w
   Matrix4X1Float[13]=Matrix4Float[iMatrixSource,4].x : Matrix4X1Float[14]=Matrix4Float[iMatrixSource,4].y : Matrix4X1Float[15]=Matrix4Float[iMatrixSource,4].z : Matrix4X1Float[16]=Matrix4Float[iMatrixSource,4].w
   Sign#=1.0 : dm#=0
   for n=1 to 4
      for di=0 to 2
         si=di+1
         for dj=0 To 2
            if dj>=n-1 then sj=dj+1 else sj=dj
            Matrix4X2Float[1+di*4+dj]=Matrix4X1Float[1+si*4+sj]
        next dj
      next di
      Determinant#=Matrix4X2Float[1]*((Matrix4X2Float[6]*Matrix4X2Float[11])-(Matrix4X2Float[7]*Matrix4X2Float[10]))
      Determinant#=Determinant#-Matrix4X2Float[2]*((Matrix4X2Float[5]*Matrix4X2Float[11])-(Matrix4X2Float[7]*Matrix4X2Float[9]))
      Determinant#=Determinant#+Matrix4X2Float[3]*((Matrix4X2Float[5]*Matrix4X2Float[10])-(Matrix4X2Float[6]*Matrix4X2Float[9]))
      dm#=dm#+(Matrix4X1Float[n]*Determinant#*Sign#)
      Sign#=-1.0*Sign#
   next n
   for i=0 to 3
      for j=0 to 3
         if ((i+j)/2.0-abs((i+j)/2))>0 then Sign#=-1.0 else Sign#=1.0
         for di=0 to 2
            for dj=0 To 2
               if di>=i then si=di+1 else si=di
               if dj>=j then sj=dj+1 else sj=dj
               Matrix4X2Float[1+di*4+dj]=Matrix4X1Float[1+si*4+sj]
           next dj
         next di
         Determinant#=Matrix4X2Float[1]*((Matrix4X2Float[6]*Matrix4X2Float[11])-(Matrix4X2Float[7]*Matrix4X2Float[10]))
         Determinant#=Determinant#-Matrix4X2Float[2]*((Matrix4X2Float[5]*Matrix4X2Float[11])-(Matrix4X2Float[7]*Matrix4X2Float[9]))
         Determinant#=Determinant#+Matrix4X2Float[3]*((Matrix4X2Float[5]*Matrix4X2Float[10])-(Matrix4X2Float[6]*Matrix4X2Float[9]))
         Matrix4X3Float[1+i+j*4]=(Determinant#*Sign#)/dm#
      next j
   next i
   Matrix4Float[iMatrixResult,1].x=Matrix4X3Float[1] : Matrix4Float[iMatrixResult,1].y=Matrix4X3Float[2] : Matrix4Float[iMatrixResult,1].z=Matrix4X3Float[3] : Matrix4Float[iMatrixResult,1].w=Matrix4X3Float[4]
   Matrix4Float[iMatrixResult,2].x=Matrix4X3Float[5] : Matrix4Float[iMatrixResult,2].y=Matrix4X3Float[6] : Matrix4Float[iMatrixResult,2].z=Matrix4X3Float[7] : Matrix4Float[iMatrixResult,2].w=Matrix4X3Float[8] 
   Matrix4Float[iMatrixResult,3].x=Matrix4X3Float[9] : Matrix4Float[iMatrixResult,3].y=Matrix4X3Float[10] : Matrix4Float[iMatrixResult,3].z=Matrix4X3Float[11] : Matrix4Float[iMatrixResult,3].w=Matrix4X3Float[12]  
   Matrix4Float[iMatrixResult,4].x=Matrix4X3Float[13] : Matrix4Float[iMatrixResult,4].y=Matrix4X3Float[14] :  Matrix4Float[iMatrixResult,4].z=Matrix4X3Float[15] : Matrix4Float[iMatrixResult,4].w=Matrix4X3Float[16]
endfunction 1

